// AnalogDeviceAudioGraph.cpp: implementation of the CAnalogDeviceAudioGraph class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "AnalogDeviceAudioGraph.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

static const GUID PROPSETID_AMEBDAD_CUSTOM_PROP = { 0xD1E5209F, 0x68FD, 0x4529, 0xBE, 0xE0, 0x5E, 0x7A, 0x1F, 0x47, 0x92, 0x00 }; // CUSTOM PROPERITY { D1E5209F-68FD-4529-BEE0-5E7A1F4792XX }

VOID CAnalogDeviceAudioGraph::SC_DEBUG( CHAR * fmt, ... )
{
	CHAR pszDebugDumpMessage[ MAX_PATH ] = "[SC510] [00000000] ";

	va_list marker;

	va_start( marker, fmt );

	StringCbPrintfA( pszDebugDumpMessage + 8, sizeof(pszDebugDumpMessage), "[%08X] ", this );

	StringCbVPrintfA( pszDebugDumpMessage + 8 + 1 + 10, sizeof(pszDebugDumpMessage), fmt, marker );

	va_end( marker );

	StringCbCatA( pszDebugDumpMessage, sizeof(pszDebugDumpMessage), "\n" );

	OutputDebugString( pszDebugDumpMessage );
}

CAnalogDeviceAudioGraph::CAnalogDeviceAudioGraph()
{
//	SC_DEBUG( "CAnalogDeviceAudioGraph::CAnalogDeviceAudioGraph()" );

	m_pCaptureSourceBaseFilter = NULL;

	m_pSampleGrabberBaseFilter = NULL;

	m_pKsPropertySet = NULL;

	m_sKsPropertySetGUID = GUID_NULL;

	m_pAMStreamConfig = NULL;

	m_pAMCrossbar = NULL;

	m_pSampleGrabber = NULL;

	m_oSampleGrabber.SetCallback( NULL, NULL );

	m_nChannels = 0;

	m_nBitsPerSample = 0;

	m_nSamplesPerSec = 0;
}

CAnalogDeviceAudioGraph::~CAnalogDeviceAudioGraph()
{
//	SC_DEBUG( "CAnalogDeviceAudioGraph::~CAnalogDeviceAudioGraph()" );

	CAnalogDeviceAudioGraph::Close( FALSE );
}

BOOL CAnalogDeviceAudioGraph::Close( BOOL bDumpDebugMessage )
{
	if( bDumpDebugMessage ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Close()" ); }

	if( m_pCommonMediaControl ) { if( GetState() == State_Running ) { Stop(); } }

	if( m_pCommonMediaControl ) { if( GetState() == State_Paused ) { Stop(); } }

	RELEASE( m_pKsPropertySet );

	RELEASE( m_pAMStreamConfig );

	RELEASE( m_pAMCrossbar );

	RELEASE( m_pSampleGrabber );

	RELEASE( m_pCaptureSourceBaseFilter );

	RELEASE( m_pSampleGrabberBaseFilter );

	m_oSampleGrabber.SetCallback( NULL, NULL );

	m_sKsPropertySetGUID = GUID_NULL;

	m_nChannels = 0;

	m_nBitsPerSample = 0;

	m_nSamplesPerSec = 0;

	return CloseGraphBuilder( bDumpDebugMessage );
}

BOOL CAnalogDeviceAudioGraph::Create( PWSTR              pwszDevName, 
									 
									  UINT               iDevNum,
									  
									  PF_BUFFER_CALLBACK pBufferCB, 
									  
									  PVOID              pUserData )
{
	WCHAR pwszFilterNameBuffer[ 256 ];

	StringCchPrintfW( pwszFilterNameBuffer, 256, L"%ws", pwszDevName );

	// CREATE GRAPH BUILDER RESOURCE
	//
	if( FALSE == CreateGraphBuilder() ) return FALSE; 

	// FIND/ADD CAPTURE SOURCE FILTER INTO GRAPH
	//
	if( FALSE == FindFilter( &AM_KSCATEGORY_CAPTURE, pwszFilterNameBuffer, iDevNum, &m_pCaptureSourceBaseFilter, pwszFilterNameBuffer, 256 ) ) return FALSE;

	if( FALSE == AddFilter( m_pCaptureSourceBaseFilter, pwszFilterNameBuffer ) ) return FALSE;

	// FIND/ADD SAMPLE GRABBER FILTER INTO GRAPH
	//
	if( FAILED(CoCreateInstance( CLSID_SampleGrabber, NULL, CLSCTX_INPROC_SERVER, IID_IBaseFilter, (VOID **)(&m_pSampleGrabberBaseFilter) )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( CoCreateInstance(CLSID_SampleGrabber) ) ERROR!!" ); return FALSE; }

	if( FALSE == AddFilter( m_pSampleGrabberBaseFilter, L"Sample Grabber" ) ) return FALSE;

	if( SUCCEEDED(m_pCommonCaptureGraphBuilder2->FindInterface( NULL, NULL, m_pSampleGrabberBaseFilter, IID_ISampleGrabber, (VOID **)(&m_pSampleGrabber) )) ) { 
		
		SC_DEBUG( "CAnalogDeviceAudioGraph::Create( FindInterface(IID_ISampleGrabber) )" ); 

		if( FAILED(m_pSampleGrabber->SetOneShot( FALSE )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( ISampleGrabber::SetOneShot() ) ERROR!!" ); return FALSE; }

		if( FAILED(m_pSampleGrabber->SetBufferSamples( FALSE )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( ISampleGrabber::SetBufferSamples() ) ERROR!!" ); return FALSE; }

		if( FAILED(m_pSampleGrabber->SetCallback( &m_oSampleGrabber, 0 )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( ISampleGrabber::SetCallback() ) ERROR!!" ); return FALSE; }

		m_oSampleGrabber.SetCallback( pBufferCB, pUserData );

		m_oSampleGrabber.SetDeviceType( 0x00000001 );
	}
	if( SUCCEEDED(m_pCommonCaptureGraphBuilder2->FindInterface( NULL,                              NULL, m_pCaptureSourceBaseFilter, IID_IKsPropertySet,       (VOID **)(&m_pKsPropertySet)       )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( FindInterface(IID_IKsPropertySet) )" ); }

	if( SUCCEEDED(m_pCommonCaptureGraphBuilder2->FindInterface( &PIN_CATEGORY_CAPTURE, &MEDIATYPE_Audio, m_pCaptureSourceBaseFilter, IID_IAMStreamConfig,      (VOID **)(&m_pAMStreamConfig)      )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( FindInterface(IID_IAMStreamConfig) )" ); }

	if( SUCCEEDED(m_pCommonCaptureGraphBuilder2->FindInterface(   &LOOK_UPSTREAM_ONLY,             NULL, m_pCaptureSourceBaseFilter, IID_IAMCrossbar,          (VOID **)(&m_pAMCrossbar)		  )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::Create( FindInterface(IID_IAMCrossbar) )" ); }

	// ENUM / FIND OUT DEVICE'S CUSTOM PROPERTY GUID (0x00 ~ 0xFF)
	// 
	if( NULL != m_pKsPropertySet ) {

		GUID guid = PROPSETID_AMEBDAD_CUSTOM_PROP;

		for( ULONG i = 0 ; i < 64 ; i++ ) { 
		
			DWORD dwDeviceSerialNumber = 0x00000000;

			DWORD cbBytes = 0x00000000;

			guid.Data4[ 7 ] = (BYTE)(i);

			if( SUCCEEDED(m_pKsPropertySet->Get( guid, 0, NULL, 0, &dwDeviceSerialNumber, sizeof(ULONG), &cbBytes )) ) { 
				
				SC_DEBUG( "CAnalogDeviceAudioGraph::Create( IKsPropertySet::GetDeviceSerialNumber( 0x%08X ) )", dwDeviceSerialNumber );

				m_sKsPropertySetGUID = guid;

				break; 
			}
		}
	}

	// CONNECT BOTH
	// 
	if( FALSE == ConnectFilters( m_pCaptureSourceBaseFilter, m_pSampleGrabberBaseFilter, 1 ) ) return FALSE;

	// COMPLETE
	//
	SC_DEBUG( "CAnalogDeviceAudioGraph::Create()" );

	return TRUE;
}

BOOL CAnalogDeviceAudioGraph::GetFormat( ULONG * pChannels, ULONG * pBitsPerSample, ULONG * pSamplesPerSec )
{
	AM_MEDIA_TYPE * pmt = NULL;

	if( NULL == m_pAMStreamConfig ) { SC_DEBUG( "CAnalogDeviceAudioGraph::GetFormat( NULL ) ERROR!!" ); return FALSE; }

	if( FAILED(m_pAMStreamConfig->GetFormat( &pmt )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::GetFormat( IAMStreamConfig::GetFormat() ) ERROR!!" ); return FALSE; }

	ULONG n_channels = ((WAVEFORMATEX *)(pmt->pbFormat))->nChannels;

	ULONG n_bits_per_sample = ((WAVEFORMATEX *)(pmt->pbFormat))->wBitsPerSample;

	ULONG n_samples_per_sec = ((WAVEFORMATEX *)(pmt->pbFormat))->nSamplesPerSec;

	if( pChannels ) { *pChannels = n_channels; }

	if( pBitsPerSample ) { *pBitsPerSample = n_bits_per_sample; }

	if( pSamplesPerSec ) { *pSamplesPerSec = n_samples_per_sec; }

	SC_DEBUG( "CAnalogDeviceAudioGraph::GetFormat( %d  %d  %d )", n_channels, n_bits_per_sample, n_samples_per_sec );

	DeleteMediaType( pmt );

	pmt = NULL;

	return TRUE;
}

BOOL CAnalogDeviceAudioGraph::SetFormat( ULONG nChannels, ULONG nBitsPerSample, ULONG nSamplesPerSec )
{
	AM_MEDIA_TYPE * pmt = NULL;

	OAFilterState eState = GetState();

	if( FALSE == Stop() ) { return FALSE; }

	if( NULL == m_pAMStreamConfig ) { SC_DEBUG( "CAnalogDeviceAudioGraph::SetFormat( NULL ) ERROR!!" ); return FALSE; }

	if( FAILED(m_pAMStreamConfig->GetFormat( &pmt )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::SetFormat( IAMStreamConfig::GetFormat() ) ERROR!!" ); return FALSE; }

	((WAVEFORMATEX *)(pmt->pbFormat))->nChannels = (USHORT)(nChannels);

	((WAVEFORMATEX *)(pmt->pbFormat))->wBitsPerSample = (USHORT)(nBitsPerSample);

	((WAVEFORMATEX *)(pmt->pbFormat))->nSamplesPerSec = (ULONG)(nSamplesPerSec);

	((WAVEFORMATEX *)(pmt->pbFormat))->nBlockAlign = (USHORT)(nChannels * nBitsPerSample / 8);

	((WAVEFORMATEX *)(pmt->pbFormat))->nAvgBytesPerSec = (ULONG)(nChannels * nBitsPerSample * nSamplesPerSec / 8);

	// DISCONNECT FILTERS
	//
	if( FALSE == DisconnectFilters( m_pCaptureSourceBaseFilter, m_pSampleGrabberBaseFilter ) ) { ; }

	// CHANGE FORMAT
	//
	if( FAILED(m_pAMStreamConfig->SetFormat( pmt )) ) { 
		
		SC_DEBUG( "CAnalogDeviceAudioGraph::SetFormat( IAMStreamConfig::SetFormat() ) ERROR!!" );
		
		goto EXIT_SETFORMAT;
	}

	// CONNECT FILTERS
	//
	if( FALSE == ConnectFilters( m_pCaptureSourceBaseFilter, m_pSampleGrabberBaseFilter, 1 ) ) goto EXIT_SETFORMAT;

	if( m_pAMCrossbar ) { // FIX CX2581 PCI MUTE BUG

		m_pAMCrossbar->Route( 1, 2 );
	}
	SC_DEBUG( "CAnalogDeviceAudioGraph::SetFormat( %d  %d  %d )", nChannels, nBitsPerSample, nSamplesPerSec );

	DeleteMediaType( pmt );

	pmt = NULL;

	if( FALSE == GetFormat( NULL, NULL, NULL ) ) { return FALSE; }

	if( eState == State_Running ) {

		return Run();
	}
	return TRUE;

EXIT_SETFORMAT:

	DeleteMediaType( pmt );

	pmt = NULL;

	return FALSE;
}

BOOL CAnalogDeviceAudioGraph::GetCustomProperty( ULONG nProperty, ULONG * pValue )
{
	DWORD cbBytes = 0x00000000;

	if( NULL == pValue ) { SC_DEBUG( "CAnalogDeviceAudioGraph::GetCustomProperty( NULL ) ERROR!!" ); return FALSE; }

	if( NULL == m_pKsPropertySet ) { SC_DEBUG( "CAnalogDeviceAudioGraph::GetCustomProperty( NULL ) ERROR!!" ); return FALSE; }

	if( FAILED(m_pKsPropertySet->Get( m_sKsPropertySetGUID, nProperty, NULL, 0, pValue, sizeof(ULONG), &cbBytes )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::GetCustomProperty( IKsPropertySet::Get() ) ERROR!!" ); return FALSE; }

//	SC_DEBUG( "CAnalogDeviceAudioGraph::GetCustomProperty( %d  %d )", nProperty, *pValue );

	return TRUE;
}

BOOL CAnalogDeviceAudioGraph::SetCustomProperty( ULONG nProperty, ULONG nValue )
{
	UINT nSize = sizeof(ULONG);

	if( NULL == m_pKsPropertySet ) { SC_DEBUG( "CAnalogDeviceAudioGraph::SetCustomProperty( NULL ) ERROR!!" ); return FALSE; }

	if( FAILED(m_pKsPropertySet->Set( m_sKsPropertySetGUID, nProperty, NULL, 0, &nValue, nSize )) ) { SC_DEBUG( "CAnalogDeviceAudioGraph::SetCustomProperty( IKsPropertySet::Set() ) ERROR!!" ); return FALSE; }

//	SC_DEBUG( "CAnalogDeviceAudioGraph::SetCustomProperty( %d  %d )", nProperty, nValue );

	return TRUE;
}
