/*
 ***************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 *
 * (c) Copyright 2002-2006, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ***************************************************************************

	Module Name:
	rtusb_io.c

	Abstract:

	Revision History:
	Who         When          What
	--------    ----------    ----------------------------------------------
	Name		Date	    Modification logs
	Paul Lin    06-25-2004    created
*/

#include	"rt_config.h"

/*
	========================================================================

	Routine Description: NIC initialization complete

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/

NTSTATUS	RTUSBFirmwareRun(
	IN	PRTMP_ADAPTER	pAd)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x01,
		0x8,
		0,
		NULL,
		0);

	return Status;
}

/*
	========================================================================

	Routine Description: Read various length data from RT2573

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBMultiRead(
	IN	PRTMP_ADAPTER	pAd,
	IN	USHORT			Offset,
	OUT	PUCHAR			pData,
	IN	USHORT			length)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		(USBD_TRANSFER_DIRECTION_IN | USBD_SHORT_TRANSFER_OK),
		DEVICE_VENDOR_REQUEST_IN,
		0x7,
		0,
		Offset,
		pData,
		length);

	return Status;
}

/*
	========================================================================

	Routine Description: Write various length data to RT2573

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBMultiWrite(
	IN	PRTMP_ADAPTER	pAd,
	IN	USHORT			Offset,
	IN	PUCHAR			pData,
	IN	USHORT			length)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x6,
		0,
		Offset,
		pData,
		length);

	return Status;
}

/*
	========================================================================

	Routine Description: Read 32-bit MAC register

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBReadMACRegister(
	IN	PRTMP_ADAPTER	pAd,
	IN	USHORT			Offset,
	OUT	PULONG			pValue)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		(USBD_TRANSFER_DIRECTION_IN | USBD_SHORT_TRANSFER_OK),
		DEVICE_VENDOR_REQUEST_IN,
		0x7,
		0,
		Offset,
		pValue,
		4);

	return Status;
}

/*
	========================================================================

	Routine Description: Write 32-bit MAC register

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBWriteMACRegister(
	IN	PRTMP_ADAPTER	pAd,
	IN	USHORT			Offset,
	IN	ULONG			Value)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x6,
		0x00,
		Offset,
		&Value,
		4);

	return Status;
}

/*
	========================================================================

	Routine Description: Write 32-bit MAC register

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS RTUSBSetLED(
	IN	PRTMP_ADAPTER		pAd,
	IN	MCU_LEDCS_STRUC		LedStatus,
	IN	USHORT				LedIndicatorStrength)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x0a,
		LedStatus.word,
		LedIndicatorStrength,
		NULL,
		0);

	return Status;
}

/*
	========================================================================

	Routine Description: Read 8-bit BBP register

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBReadBBPRegister(
	IN	PRTMP_ADAPTER	pAd,
	IN	UCHAR			Id,
	IN	PUCHAR			pValue)
{
	PHY_CSR3_STRUC	PhyCsr3;
	UINT			i = 0;

	// Verify the busy condition
	do
	{
		RTUSBReadMACRegister(pAd, PHY_CSR3, &PhyCsr3.word);
		if (!(PhyCsr3.field.Busy == BUSY))
			break;
		i++;
	}
	while ((i < RETRY_LIMIT) && (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)));

	if ((i == RETRY_LIMIT) || (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
	{
		//
		// Read failed then Return Default value.
		//
		*pValue = pAd->BbpWriteLatch[Id];

		DBGPRINT_RAW(RT_DEBUG_ERROR, ("Retry count exhausted or device removed!!!\n"));
		return STATUS_UNSUCCESSFUL;
	}

	// Prepare for write material
	PhyCsr3.word 				= 0;
	PhyCsr3.field.fRead			= 1;
	PhyCsr3.field.Busy			= 1;
	PhyCsr3.field.RegNum 		= Id;
	RTUSBWriteMACRegister(pAd, PHY_CSR3, PhyCsr3.word);

	i = 0;
	// Verify the busy condition
	do
	{
		RTUSBReadMACRegister(pAd, PHY_CSR3, &PhyCsr3.word);
		if (!(PhyCsr3.field.Busy == BUSY))
		{
			*pValue = (UCHAR)PhyCsr3.field.Value;
			break;
		}
		i++;
	}
	while ((i < RETRY_LIMIT) && (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)));

	if ((i == RETRY_LIMIT) || (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
	{
		//
		// Read failed then Return Default value.
		//
		*pValue = pAd->BbpWriteLatch[Id];

		DBGPRINT_RAW(RT_DEBUG_ERROR, ("Retry count exhausted or device removed!!!\n"));
		return STATUS_UNSUCCESSFUL;
	}

	return STATUS_SUCCESS;
}

/*
	========================================================================

	Routine Description: Write 8-bit BBP register

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBWriteBBPRegister(
	IN	PRTMP_ADAPTER	pAd,
	IN	UCHAR			Id,
	IN	UCHAR			Value)
{
	PHY_CSR3_STRUC	PhyCsr3;
	UINT			i = 0;

	// Verify the busy condition
	do
	{
		RTUSBReadMACRegister(pAd, PHY_CSR3, &PhyCsr3.word);
		if (!(PhyCsr3.field.Busy == BUSY))
			break;
		i++;
	}
	while ((i < RETRY_LIMIT) && (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)));

	if ((i == RETRY_LIMIT) || (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
	{
		DBGPRINT_RAW(RT_DEBUG_ERROR, ("Retry count exhausted or device removed!!!\n"));
		return STATUS_UNSUCCESSFUL;
	}

	// Prepare for write material
	PhyCsr3.word 				= 0;
	PhyCsr3.field.fRead			= 0;
	PhyCsr3.field.Value			= Value;
	PhyCsr3.field.Busy			= 1;
	PhyCsr3.field.RegNum 		= Id;
	RTUSBWriteMACRegister(pAd, PHY_CSR3, PhyCsr3.word);

	pAd->BbpWriteLatch[Id] = Value;

	return STATUS_SUCCESS;
}

/*
	========================================================================

	Routine Description: Write RF register through MAC

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBWriteRFRegister(
	IN	PRTMP_ADAPTER	pAd,
	IN	ULONG			Value)
{
	PHY_CSR4_STRUC	PhyCsr4;
	UINT			i = 0;

	do
	{
		RTUSBReadMACRegister(pAd, PHY_CSR4, &PhyCsr4.word);
		if (!(PhyCsr4.field.Busy))
			break;
		i++;
	}
	while ((i < RETRY_LIMIT) && (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)));

	if ((i == RETRY_LIMIT) || (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
	{
		DBGPRINT_RAW(RT_DEBUG_ERROR, ("Retry count exhausted or device removed!!!\n"));
		return STATUS_UNSUCCESSFUL;
	}

	RTUSBWriteMACRegister(pAd, PHY_CSR4, Value);

	return STATUS_SUCCESS;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBReadEEPROM(
	IN	PRTMP_ADAPTER	pAd,
	IN	USHORT			Offset,
	OUT	PUCHAR			pData,
	IN	USHORT			length)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		(USBD_TRANSFER_DIRECTION_IN | USBD_SHORT_TRANSFER_OK),
		DEVICE_VENDOR_REQUEST_IN,
		0x9,
		0,
		Offset,
		pData,
		length);

	return Status;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSBWriteEEPROM(
	IN	PRTMP_ADAPTER	pAd,
	IN	USHORT			Offset,
	IN	PUCHAR			pData,
	IN	USHORT			length)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x8,
		0,
		Offset,
		pData,
		length);

	return Status;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS RTUSBPutToSleep(
	IN	PRTMP_ADAPTER	pAd)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x01,
		0x07,
		0,
		NULL,
		0);

	return Status;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS RTUSBWakeUp(
	IN	PRTMP_ADAPTER	pAd)
{
	NTSTATUS	Status;

	Status = RTUSB_VendorRequest(
		pAd,
		USBD_TRANSFER_DIRECTION_OUT,
		DEVICE_VENDOR_REQUEST_OUT,
		0x01,
		0x09,
		0,
		NULL,
		0);

	return Status;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBInitializeCmdQ(
	IN	PCmdQ	cmdq)
{
	cmdq->head = NULL;
	cmdq->tail = NULL;
	cmdq->size = 0;
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NDIS_STATUS	RTUSBEnqueueCmdFromNdis(
	IN	PRTMP_ADAPTER	pAd,
	IN	NDIS_OID		Oid,
	IN	BOOLEAN			SetInformation,
	IN	PVOID			pInformationBuffer,
	IN	ULONG			InformationBufferLength)
{
	NDIS_STATUS	status;
	unsigned long IrqFlags;
	PCmdQElmt	cmdqelmt = NULL;

	if (pAd->RTUSBCmdThr_pid < 0)
		return (NDIS_STATUS_RESOURCES);

	status = RTMPAllocateMemory((PVOID*)&cmdqelmt, sizeof(CmdQElmt));
	if ((status != NDIS_STATUS_SUCCESS) || (cmdqelmt == NULL))
	{
		DBGPRINT(RT_DEBUG_ERROR,
			("RTUSBEnqueueCmdFromNdis, cmdqelmt allocate Memory size(=%d) failed\n", sizeof(CmdQElmt)));
		return (NDIS_STATUS_RESOURCES);
	}

	if ((Oid == RT_OID_MULTI_READ_MAC) ||
		(Oid == RT_OID_VENDOR_READ_BBP) ||
		(Oid == RT_OID_802_11_QUERY_HARDWARE_REGISTER) ||
		(Oid == RT_OID_USB_VENDOR_EEPROM_READ))
	{
		cmdqelmt->buffer = pInformationBuffer;
	}
	else
	{
		cmdqelmt->buffer = NULL;
		if (pInformationBuffer != NULL)
		{
			status = RTMPAllocateMemory((PVOID*)&(cmdqelmt->buffer), InformationBufferLength);
			if ((status != NDIS_STATUS_SUCCESS) || (cmdqelmt->buffer == NULL))
			{
				if (cmdqelmt)
					NdisFreeMemory(cmdqelmt, sizeof(CmdQElmt), 0);

				DBGPRINT(RT_DEBUG_ERROR,
					("RTUSBEnqueueCmdFromNdis, cmdqelmt->buffer allocate Memory size(=%d) failed\n", InformationBufferLength));
				return (NDIS_STATUS_RESOURCES);
			}
			else
			{
				NdisMoveMemory(cmdqelmt->buffer, pInformationBuffer, InformationBufferLength);
				cmdqelmt->bufferlength = InformationBufferLength;
			}
		}
		else
			cmdqelmt->bufferlength = 0;
	}

	cmdqelmt->command = Oid;
	cmdqelmt->CmdFromNdis = TRUE;
	if (SetInformation == TRUE)
		cmdqelmt->SetOperation = TRUE;
	else
		cmdqelmt->SetOperation = FALSE;

	NdisAcquireSpinLock(&pAd->CmdQLock, IrqFlags);
	EnqueueCmd((&pAd->CmdQ), cmdqelmt);
	NdisReleaseSpinLock(&pAd->CmdQLock, IrqFlags);

    RTUSBCMDUp(pAd);

#ifdef WIN_NDIS
	if ((Oid != OID_802_11_ADD_KEY) && (SetInformation == TRUE))
	{
		return(NDIS_STATUS_SUCCESS);
	}
	else
	{
		return(NDIS_STATUS_PENDING);
	}
#else
    return(NDIS_STATUS_SUCCESS);
#endif
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
#ifdef WIN_NDIS
NDIS_STATUS RTUSBEnqueueInternalCmd(
	IN PRTMP_ADAPTER	pAd,
	IN NDIS_OID			Oid,
	IN PVOID			pInformationBuffer,
	IN ULONG			InformationBufferLength)
{
	NDIS_STATUS	status;
	PCmdQElmt	cmdqelmt = NULL;

	DBGPRINT(RT_DEBUG_INFO, ("--->RTUSBEnqueueInternalCmd\n"));

	status = RTMPAllocateMemory(&cmdqelmt, sizeof(CmdQElmt));
	if ((status != NDIS_STATUS_SUCCESS) || (cmdqelmt == NULL))
		return (NDIS_STATUS_RESOURCES);

	NdisZeroMemory(cmdqelmt, sizeof(CmdQElmt));

	if(InformationBufferLength > 0)
	{
		status = RTMPAllocateMemory(&(cmdqelmt->buffer), InformationBufferLength);

		if ((status != NDIS_STATUS_SUCCESS) || (cmdqelmt->buffer == NULL))
		{
			NdisFreeMemory(cmdqelmt, sizeof(CmdQElmt), 0);
			return (NDIS_STATUS_RESOURCES);
		}
		else
		{
			NdisMoveMemory(cmdqelmt->buffer, pInformationBuffer, InformationBufferLength);
			cmdqelmt->bufferlength = InformationBufferLength;
		}
	}
	else
	{
		cmdqelmt->buffer = NULL;
		cmdqelmt->bufferlength = 0;
	}

	cmdqelmt->command = Oid;
	cmdqelmt->CmdFromNdis = FALSE;

	if (cmdqelmt != NULL)
	{
		NdisAcquireSpinLock(&pAd->CmdQLock);
		EnqueueCmd((&pAd->CmdQ), cmdqelmt);
		NdisReleaseSpinLock(&pAd->CmdQLock);

		KeSetEvent(&pAd->ControlEvent, 0, FALSE);
	}

	DBGPRINT(RT_DEBUG_INFO, ("<---RTUSBEnqueueInternalCmd\n"));
	return(NDIS_STATUS_SUCCESS);
}
#else // WIN_NDIS //
NDIS_STATUS RTUSBEnqueueInternalCmd(
	IN	PRTMP_ADAPTER	pAd,
	IN	NDIS_OID		Oid,
	IN PVOID			pInformationBuffer,
	IN ULONG			InformationBufferLength)
{
	PCmdQElmt	cmdqelmt = NULL;
    unsigned long IrqFlags;

    if (pAd->RTUSBCmdThr_pid < 0)
		return NDIS_STATUS_RESOURCES;

	switch (Oid)
	{
		case RT_OID_CHECK_GPIO:
			cmdqelmt = &(pAd->CmdQElements[CMD_CHECK_GPIO]);
			break;

		case RT_OID_PERIODIC_EXECUT:
			cmdqelmt = &(pAd->CmdQElements[CMD_PERIODIC_EXECUT]);
			break;

		//For Alpha only
		case RT_OID_ASICLED_EXECUT:
			cmdqelmt = &(pAd->CmdQElements[CMD_ASICLED_EXECUT]);
			break;

		case RT_OID_UPDATE_TX_RATE:
			cmdqelmt = &(pAd->CmdQElements[CMD_UPDATE_TX_RATE]);
			break;

		case RT_OID_SET_PSM_BIT_SAVE:
			cmdqelmt = &(pAd->CmdQElements[CMD_SET_PSM_SAVE]);
			break;

		case RT_OID_LINK_DOWN:
			cmdqelmt = &(pAd->CmdQElements[CMD_LINK_DOWN]);
			break;

		case RT_OID_USB_RESET_BULK_IN:
			cmdqelmt = &(pAd->CmdQElements[CMD_RESET_BULKIN]);
			break;

		case RT_OID_USB_RESET_BULK_OUT:
			cmdqelmt = &(pAd->CmdQElements[CMD_RESET_BULKOUT]);
			break;

		case RT_OID_RESET_FROM_ERROR:
			cmdqelmt = &(pAd->CmdQElements[CMD_RESET_FROM_ERROR]);
			break;

		case RT_OID_RESET_FROM_NDIS:
			cmdqelmt = &(pAd->CmdQElements[CMD_RESET_FROM_NDIS]);
			break;

		case RT_PERFORM_SOFT_DIVERSITY:
			cmdqelmt = &(pAd->CmdQElements[CMD_SOFT_DIVERSITY]);
			break;

        case RT_OID_FORCE_WAKE_UP:
            cmdqelmt = &(pAd->CmdQElements[CMD_FORCE_WAKEUP]);
            break;

        case RT_OID_SET_PSM_BIT_ACTIVE:
            cmdqelmt = &(pAd->CmdQElements[CMD_SET_PSM_ACTIVE]);
        break;

		default:
			break;
	}

	if ((cmdqelmt != NULL) && (cmdqelmt->InUse == FALSE) && (pAd->RTUSBCmdThr_pid > 0))
	{
		cmdqelmt->InUse = TRUE;
		cmdqelmt->command = Oid;

		NdisAcquireSpinLock(&pAd->CmdQLock, IrqFlags);
		EnqueueCmd((&pAd->CmdQ), cmdqelmt);
		NdisReleaseSpinLock(&pAd->CmdQLock, IrqFlags);

		RTUSBCMDUp(pAd);
	}
	return NDIS_STATUS_SUCCESS;
}
#endif // WIN_NDIS //

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSBDequeueCmd(
	IN	PCmdQ		cmdq,
	OUT	PCmdQElmt	*pcmdqelmt)
{
	*pcmdqelmt = cmdq->head;

	if (*pcmdqelmt != NULL)
	{
		cmdq->head = cmdq->head->next;
		cmdq->size--;
		if (cmdq->size == 0)
			cmdq->tail = NULL;
	}
}

/*
	========================================================================
	  usb_control_msg - Builds a control urb, sends it off and waits for completion
	  @dev: pointer to the usb device to send the message to
	  @pipe: endpoint "pipe" to send the message to
	  @request: USB message request value
	  @requesttype: USB message request type value
	  @value: USB message value
	  @index: USB message index value
	  @data: pointer to the data to send
	  @size: length in bytes of the data to send
	  @timeout: time in jiffies to wait for the message to complete before
			  timing out (if 0 the wait is forever)
	  Context: !in_interrupt ()

	  This function sends a simple control message to a specified endpoint
	  and waits for the message to complete, or timeout.
	  If successful, it returns the number of bytes transferred, otherwise a negative error number.

	 Don't use this function from within an interrupt context, like a
	  bottom half handler.	If you need an asynchronous message, or need to send
	  a message from within interrupt context, use usb_submit_urb()
	  If a thread in your driver uses this call, make sure your disconnect()
	  method can wait for it to complete.  Since you don't have a handle on
	  the URB used, you can't cancel the request.


	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
#ifdef WIN_NDIS
NTSTATUS	RTUSB_VendorRequest(
	IN	PRTMP_ADAPTER	pAd,
	IN	ULONG			TransferFlags,
	IN	UCHAR			ReservedBits,
	IN	UCHAR			Request,
	IN	USHORT			Value,
	IN	USHORT			Index,
	IN	PVOID			TransferBuffer,
	IN	ULONG			TransferBufferLength)
{
	URB			LocalUrb;
	PURB		pUrb = &LocalUrb;
	NTSTATUS	Status;
#if defined(ME_98) && (ME_98 == 1)
	ULONG		Count = 0;
#endif

	// Acquire Control token
	while (1)
	{
		NdisAcquireSpinLock(&pAd->ControlLock);
		if (pAd->ControlPending == FALSE)
		{
			pAd->ControlPending = TRUE;
			NdisReleaseSpinLock(&pAd->ControlLock);
			break;
		}
		else
		{
			NdisReleaseSpinLock(&pAd->ControlLock);

			if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
				return (STATUS_DEVICE_NOT_CONNECTED);

			NdisMSleep(50);
#if defined(ME_98) && (ME_98 == 1)
			//
			// IRP no response more than 500 milliscond, means other VendorRequst can't be service.
			// To prevent this, reset the pAd->ControlPending flag.
			// TO:
			//	  Find the root cause.
			//
			Count++;
			if (Count >= 1000)
			{
				pAd->ControlPending = FALSE;
			}
#endif
		}
	}

	// Build vendor request URB
	UsbBuildVendorRequest(
		pUrb,
		URB_FUNCTION_VENDOR_DEVICE,
		sizeof(struct _URB_CONTROL_VENDOR_OR_CLASS_REQUEST),
		TransferFlags,
		ReservedBits,
		Request,
		Value,
		Index,
		TransferBuffer,
		NULL,
		TransferBufferLength,
		NULL);

	// Call underline USBD driver
	Status = RTUSB_CallUSBD(pAd, pUrb);

	// Release token
	NdisAcquireSpinLock(&pAd->ControlLock);
	pAd->ControlPending = FALSE;
	NdisReleaseSpinLock(&pAd->ControlLock);

	return Status;
}
#else // WIN_NDIS //
INT	    RTUSB_VendorRequest(
	IN	PRTMP_ADAPTER	pAd,
	IN	ULONG			TransferFlags,
	IN	UCHAR			RequestType,
	IN	UCHAR			Request,
	IN	USHORT			Value,
	IN	USHORT			Index,
	IN	PVOID			TransferBuffer,
	IN	ULONG			TransferBufferLength)
{
	int ret;

	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
	{
		DBGPRINT(RT_DEBUG_ERROR, ("device disconnected\n"));
		return -1;
	}
	else if (in_interrupt())
	{
		DBGPRINT(RT_DEBUG_ERROR, ("in_interrupt, return RTUSB_VendorRequest\n"));

		return -1;
	}
	else
	{

		if( RequestType == DEVICE_VENDOR_REQUEST_OUT)
			ret=usb_control_msg(pAd->pUsb_Dev, usb_sndctrlpipe( pAd->pUsb_Dev, 0 ), Request, RequestType, Value,Index, TransferBuffer, TransferBufferLength, CONTROL_TIMEOUT_JIFFIES);
		else if(RequestType == DEVICE_VENDOR_REQUEST_IN)
			ret=usb_control_msg(pAd->pUsb_Dev, usb_rcvctrlpipe( pAd->pUsb_Dev, 0 ), Request, RequestType, Value,Index, TransferBuffer, TransferBufferLength, CONTROL_TIMEOUT_JIFFIES);
		else
		{
			DBGPRINT(RT_DEBUG_ERROR, ("vendor request direction is failed\n"));
			ret = -1;
		}

        if (ret < 0)
			DBGPRINT(RT_DEBUG_ERROR, ("USBVendorRequest failed ret=%d \n",ret));

#if 0
        // retry
		if (ret < 0) {
			int temp_i=0;
			DBGPRINT(RT_DEBUG_ERROR, ("USBVendorRequest failed ret=%d, \n",ret));
			ret = 0;
			do
			{
				if( RequestType == DEVICE_VENDOR_REQUEST_OUT)
					ret=usb_control_msg(pAd->pUsb_Dev, usb_sndctrlpipe( pAd->pUsb_Dev, 0 ), Request, RequestType, Value,Index, TransferBuffer, TransferBufferLength, CONTROL_TIMEOUT_JIFFIES);
				else if(RequestType == DEVICE_VENDOR_REQUEST_IN)
					ret=usb_control_msg(pAd->pUsb_Dev, usb_rcvctrlpipe( pAd->pUsb_Dev, 0 ), Request, RequestType, Value,Index, TransferBuffer, TransferBufferLength, CONTROL_TIMEOUT_JIFFIES);
				temp_i++;
			} while( (ret < 0) && (temp_i <= 1) );

			if( ret >= 0)
				return ret;

		}
#endif

	}
	return ret;
}
#endif // WIN_NDIS //

#ifdef WIN_NDIS
/*
	========================================================================

	Routine Description:
		Creates an IRP to submite a URB to USBD synchronously. Callers of
		this function must be running at PASSIVE LEVEL

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_CallUSBD(
	IN	PRTMP_ADAPTER	pAd,
	IN	PURB			pUrb)
{
	NTSTATUS			ntStatus = STATUS_SUCCESS;
	PIRP				pIrp = NULL;
	KEVENT				IoEvent;
	IO_STATUS_BLOCK		IoStatus;
	BOOLEAN				IoWaitStatus;
	PIO_STACK_LOCATION	pNextStack;

	DBGPRINT_RAW(RT_DEBUG_INFO, ("--->RTUSB_CallUSBD\n"));

	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
		return (STATUS_DEVICE_NOT_CONNECTED);

	// Check for correct level
	ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

	// Initialize event
	NdisInitializeEvent((PNDIS_EVENT)&IoEvent);

	// Build an IRP in which to pass the URB to USBD
	pIrp = IoBuildDeviceIoControlRequest(
				IOCTL_INTERNAL_USB_SUBMIT_URB,
				pAd->pNextDeviceObject,	// Points to the next-lower
				NULL,							// No Input
				0,
				NULL,							// No output
				0,
				TRUE,							// Internal use IRP_MJ_INTERNAL_DEVICE_CONTROL
				&IoEvent,
				&IoStatus);

	if (!pIrp)
	{
		DBGPRINT_RAW(RT_DEBUG_ERROR, ("RTUSB_CallUSBD::IoBuildDeviceIoControlRequest failed irp=NULL\n"));
		ntStatus = STATUS_INSUFFICIENT_RESOURCES;
		return(ntStatus);
	}

	// Pass the URB to the USB driver stack
	pNextStack = IoGetNextIrpStackLocation(pIrp);
	ASSERT(pNextStack);
	pNextStack->Parameters.Others.Argument1 = pUrb;

	ntStatus = IoCallDriver(pAd->pNextDeviceObject, pIrp);
	if (ntStatus == STATUS_PENDING)
	{
		// If the request is pending, wait until it completes
		DBGPRINT_RAW(RT_DEBUG_INFO,("IoCallDriver PENDING Irp=0x%X\n", pIrp));

		IoWaitStatus = NdisWaitEvent((PNDIS_EVENT)&IoEvent, 0);

		if (!IoWaitStatus)
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR,("OSWaitEvent STATUS_TIMEOUT\n"));
			ntStatus = STATUS_TIMEOUT;
			return(ntStatus);
		}
	}
	else
	{
		IoStatus.Status = ntStatus;
	}

	// USBD maps the error code for us
	ntStatus = IoStatus.Status;

	// If the NT status is successful, check the URB status
	if (NT_SUCCESS(ntStatus))
	{
		if (!USBD_SUCCESS(URB_STATUS(pUrb)))
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR,("URB Status=0x%X\n",URB_STATUS(pUrb)));
			ntStatus = STATUS_UNSUCCESSFUL;
			return(ntStatus);
		}
	}
	else if ((ntStatus == STATUS_DEVICE_NOT_CONNECTED) ||
			 (ntStatus == STATUS_DEVICE_REMOVED))
	{
		//
		// surprise remove.
		//
		RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST);
		DBGPRINT_RAW(RT_DEBUG_ERROR,("IoCallDriver FAILED IoStatus=0x%08x, URB_STATUS=0x%08x\n",ntStatus, URB_STATUS(pUrb)));
	}
	else
	{
		DBGPRINT_RAW(RT_DEBUG_ERROR,("IoCallDriver FAILED IoStatus=0x%08x, URB_STATUS=0x%08x\n",ntStatus, URB_STATUS(pUrb)));
	}

	return(ntStatus);
}

/*
	========================================================================

	Routine Description:
		Creates and submits a URB to USBD to get the Device Descriptor

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_GetDeviceDescriptor(
	IN	PRTMP_ADAPTER	pAd)
{
	URB			LocalUrb;
	PURB		pUrb = &LocalUrb;
	NTSTATUS	Status = STATUS_SUCCESS;
	ULONG		size;

	// Allocate device descriptor memory
	size = sizeof(USB_DEVICE_DESCRIPTOR);
	RTMPAllocateMemory(&pAd->pUsbDeviceDescriptor, size);
	if (pAd->pUsbDeviceDescriptor == NULL)
	{
		Status = STATUS_INSUFFICIENT_RESOURCES;
		return(Status);
	}

	// Build the Get Configuration Descriptor request and submit to USBD
	UsbBuildGetDescriptorRequest(
			pUrb,
			sizeof (struct _URB_CONTROL_DESCRIPTOR_REQUEST),
			USB_DEVICE_DESCRIPTOR_TYPE,
			0,
			0,
			pAd->pUsbDeviceDescriptor,
			NULL,
			size,
			NULL);

	Status = RTUSB_CallUSBD(pAd, pUrb);
	if (!NT_SUCCESS(Status))
	{
		NdisFreeMemory(pAd->pUsbDeviceDescriptor, size, 0);
		pAd->pUsbDeviceDescriptor = NULL;
	}

	return(Status);
}

/*
	========================================================================

	Routine Description:
		Free Device Descriptor

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSB_FreeDeviceDescriptor(
	IN	PRTMP_ADAPTER	pAd)
{
	if (pAd->pUsbDeviceDescriptor)
	{
		NdisFreeMemory(pAd->pUsbDeviceDescriptor, sizeof(USB_DEVICE_DESCRIPTOR), 0);
		pAd->pUsbDeviceDescriptor = NULL;
	}
}

/*
	========================================================================

	Routine Description:
		Free Config Descriptor

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSB_FreeConfigDescriptor(
	IN	PUSB_CONFIGURATION_DESCRIPTOR	pConfigDesc)
{
	if (pConfigDesc)
	{
		NdisFreeMemory(pConfigDesc, USB_DEVICE_MAX_CONFIG_DESCRIPTOR_SIZE, 0);
	}
}

/*
	========================================================================

	Routine Description:
		Allocate Config Descriptor

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
PUSB_CONFIGURATION_DESCRIPTOR	RTUSB_GetConfigDescriptor(
	IN	PRTMP_ADAPTER	pAd)
{
	PUSB_CONFIGURATION_DESCRIPTOR	pConfigDesc = NULL;
	URB								LocalUrb;
	PURB							pUrb = &LocalUrb;
	ULONG							DescSize;
	NTSTATUS						Status;

	DescSize = sizeof(USB_CONFIGURATION_DESCRIPTOR);

	// When we retrieve the configuration descriptor,
	// all interface, endpoint, class-specific, and vendor-specific descriptors
	// for the configuration also are retrieved.
	// Initially we don't know the size of the structure to allocate the buffer
	// just do a good guess and compare with pConfigDesc->wTotalLength
	// If allocated size is less than wTotalLength then retry with newsize.
	while (1)
	{
		RTMPAllocateMemory(&pConfigDesc, DescSize);
		if (pConfigDesc == NULL)
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR, ("HostAllocateMemory pConfigDesc failed size=%d\n", DescSize));
			break;
		}

		// Build the Get Configuration Descriptor request and submit to USBD
		UsbBuildGetDescriptorRequest(
				pUrb,
				sizeof(struct _URB_CONTROL_DESCRIPTOR_REQUEST),
				USB_CONFIGURATION_DESCRIPTOR_TYPE,
				0,
				0,
				pConfigDesc,
				NULL,
				DescSize,
				NULL);

		Status = RTUSB_CallUSBD(pAd, pUrb);
		if (!NT_SUCCESS(Status))
		{
			RTUSB_FreeConfigDescriptor(pConfigDesc);
			pConfigDesc = NULL;
			break;
		}

		// If the descriptor is larger than we allocated, increase the size
		// and retry.
		if ((pUrb->UrbControlDescriptorRequest.TransferBufferLength > 0) &&
			(pConfigDesc->wTotalLength > DescSize))
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR, ("TransferBufferLength=%d wTotalLength=%d>%d\n",
				pUrb->UrbControlDescriptorRequest.TransferBufferLength,
				pConfigDesc->wTotalLength,
				DescSize));

			DescSize = pConfigDesc->wTotalLength;
			RTUSB_FreeConfigDescriptor(pConfigDesc);
			pConfigDesc = NULL;
		}
		else
		{
			// Allocated buffer is enough. break the retry loop.
			break;
		}
	}

	return(pConfigDesc);
}

/*
	========================================================================

	Routine Description:
		Free Interface list

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSB_FreeInterfaceList(
	IN	PUSBD_INTERFACE_LIST_ENTRY	pInterfaceList,
	IN	ULONG						NumInterfaces)
{
	if (pInterfaceList)
	{
		NdisFreeMemory(pInterfaceList, sizeof(USBD_INTERFACE_LIST_ENTRY) * (NumInterfaces + 1), 0);
	}
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
PUSBD_INTERFACE_LIST_ENTRY				RTUSB_GetInterfaceList(
	IN	PRTMP_ADAPTER					pAd,
	IN	PUSB_CONFIGURATION_DESCRIPTOR	pConfigDesc)
{
	PUSB_INTERFACE_DESCRIPTOR	pInterfaceDesc = NULL;
	PUSBD_INTERFACE_LIST_ENTRY	pInterfaceList = NULL;
	ULONG						ListSize;

	while (1)
	{
		// Allocate memory for the interface list.
		ListSize = sizeof(USBD_INTERFACE_LIST_ENTRY) * (pConfigDesc->bNumInterfaces + 1);
		RTMPAllocateMemory(&pInterfaceList, ListSize);
		if (pInterfaceList == NULL)
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR, ("HostAllocateMemory pInterfaceList failed size=%d\n", ListSize));
			break;
		}

		// Parse the configuration descriptor to find the interface descriptor.
		pInterfaceDesc = USBD_ParseConfigurationDescriptorEx(
				pConfigDesc,
				pConfigDesc, // search from start of config descriptor
				-1,
				-1,
				-1,
				-1,
				-1);

		if (pInterfaceDesc == NULL)
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR, ("USBD_ParseConfigurationDescriptorEx failed\n"));

			RTUSB_FreeInterfaceList(pInterfaceList, pConfigDesc->bNumInterfaces);
			pInterfaceList = NULL;
			break;
		}

		pInterfaceList->InterfaceDescriptor        = pInterfaceDesc;
		pInterfaceList->Interface                  = NULL;
		(pInterfaceList + 1) ->InterfaceDescriptor = NULL;
		(pInterfaceList + 1) ->Interface           = NULL;
		break;
	}

	return (pInterfaceList);
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSB_FreeConfigUrb(
	IN	PURB	pUrb)
{
	if (pUrb)
	{
		NdisFreeMemory(pUrb, sizeof(URB), 0);
	}
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_SelectConfiguration(
	IN	PRTMP_ADAPTER					pAd,
	IN	PUSB_CONFIGURATION_DESCRIPTOR	pConfigDesc,
	IN	PUSBD_INTERFACE_LIST_ENTRY		pInterfaceList,
	OUT	PURB							*pConfigReqUrb)
{
	NTSTATUS	Status;
	PURB		pUrb;

	ASSERT(pConfigDesc);
	ASSERT(pInterfaceList);

	*pConfigReqUrb = NULL;

	// Create the URB for the configuration request.
	pUrb = USBD_CreateConfigurationRequestEx(pConfigDesc, pInterfaceList);
	if (pUrb == NULL)
	{
		DBGPRINT_RAW(RT_DEBUG_ERROR, ("USBD_CreateConfigurationRequestEx failed pUrb=NULL\n"));
		Status = STATUS_INSUFFICIENT_RESOURCES;
		return(Status);
	}

	// Pass the configuration request to USBD
	Status = RTUSB_CallUSBD(pAd, pUrb);

	if (NT_SUCCESS(Status))
	{
		// return pointer to allocated URB
		*pConfigReqUrb = pUrb;
	}
	else
	{
		RTUSB_FreeConfigUrb(pUrb);
	}

	return (Status);
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_ConfigurePipes(
	IN	PRTMP_ADAPTER				pAd,
	IN	PUSBD_INTERFACE_INFORMATION	pInterface)
{
	NTSTATUS				Status = STATUS_SUCCESS;
	PUSBD_PIPE_INFORMATION	pPipeDesc;
	ULONG					i;
	ULONG					BulkOutIdx;

	BulkOutIdx = 0;
	pAd->NumberOfPipes = pInterface->NumberOfPipes;
	DBGPRINT(RT_DEBUG_TRACE, ("RTUSB_ConfigurePipes, NumberOfPipes(=%d)\n", pAd->NumberOfPipes));
	for (i = 0; i < pInterface->NumberOfPipes; i++)//save information of each pipe
	{
		if ((pInterface->Pipes[i].PipeType == UsbdPipeTypeBulk) &&
			(USB_ENDPOINT_DIRECTION_IN(pInterface->Pipes[i].EndpointAddress)))
		{
			pAd->BulkInPipeHandle    = pInterface->Pipes[i].PipeHandle;
			pAd->BulkInMaxPacketSize = pInterface->Pipes[i].MaximumPacketSize;
			DBGPRINT_RAW(RT_DEBUG_INFO, ("BULK IN MaximumPacketSize = %d\n", pAd->BulkInMaxPacketSize));
		}
		else if ((pInterface->Pipes[i].PipeType == UsbdPipeTypeBulk) &&
			(USB_ENDPOINT_DIRECTION_OUT(pInterface->Pipes[i].EndpointAddress)))
		{
			pAd->BulkOutPipeHandle[BulkOutIdx++] = pInterface->Pipes[i].PipeHandle;
			pAd->BulkOutMaxPacketSize = pInterface->Pipes[i].MaximumPacketSize;
			DBGPRINT_RAW(RT_DEBUG_INFO, ("BULK OUT MaximumPacketSize = %d\n", pAd->BulkOutMaxPacketSize));
			DBGPRINT_RAW(RT_DEBUG_INFO, ("EP address = %x  \n", pInterface->Pipes[i].EndpointAddress));
		}

	}
	return(Status);
}

/*
	========================================================================

	Routine Description:

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_ConfigureDevice(
	IN	PRTMP_ADAPTER	pAd)
{

	PUSB_CONFIGURATION_DESCRIPTOR	pConfigDesc = NULL;
	PUSBD_INTERFACE_LIST_ENTRY		pInterfaceList = NULL;
	PUSBD_INTERFACE_INFORMATION		pInterface = NULL;
	PURB							pConfigReqUrb = NULL;
	size_t							NumInterfaces = 0;
	NDIS_STATUS						Status = NDIS_STATUS_FAILURE;

	DBGPRINT_RAW(RT_DEBUG_TRACE, ("---> RTUSB_ConfigureDevice\n"));

	while (1)
	{
		// Get the Configuration descriptor
		pConfigDesc = RTUSB_GetConfigDescriptor(pAd);
		if (pConfigDesc == NULL)
		{
			break;
		}

		// If the number of interfaces provided by the device doesn't match the
		// number of interfaces we are expecting, return failure
		if (pConfigDesc->bNumInterfaces != 1)
		{
			DBGPRINT_RAW(RT_DEBUG_ERROR, ("Device has invalid number of interfaces %d\n", pConfigDesc->bNumInterfaces));
			break;
		}
		NumInterfaces = pConfigDesc->bNumInterfaces;

		// Get the interface list
		pInterfaceList = RTUSB_GetInterfaceList(pAd, pConfigDesc);
		if (pInterfaceList == NULL)
		{
			break;
		}

		// Select Configuration
		Status = RTUSB_SelectConfiguration(
					pAd,
					pConfigDesc,
					pInterfaceList,
					&pConfigReqUrb);

		// We are done with Configuration Descriptor
		if (!NT_SUCCESS(Status))
		{
			break;
		}

		// Configure the pipes
		pInterface = pInterfaceList[0].Interface;
		Status = RTUSB_ConfigurePipes(pAd, pInterface);
		break;
	}

	if (pConfigReqUrb)
	{
		RTUSB_FreeConfigUrb(pConfigReqUrb);
	}
	if (pInterfaceList)
	{
		RTUSB_FreeInterfaceList(pInterfaceList, NumInterfaces);
	}
	if (pConfigDesc)
	{
		RTUSB_FreeConfigDescriptor(pConfigDesc);
	}

	return(Status);
}

/*
	========================================================================

	Routine Description:
		Initializes some of the USB related members of the Adapter Object,
		and calls functions to get the device descriptor, to configure the
		device and to allocates IRPs/URBs

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
BOOLEAN	RTUSB_InterfaceStart(
	IN	PRTMP_ADAPTER	pAd)
{
	NTSTATUS ntStatus;

	ntStatus = RTUSB_GetDeviceDescriptor(pAd);
	if (!NT_SUCCESS(ntStatus))
	{
		return FALSE;
	}

	ntStatus = RTUSB_ConfigureDevice(pAd);
	if (!NT_SUCCESS(ntStatus))
	{
		return FALSE;
	}

	return TRUE;
}

/*
	========================================================================

	Routine Description:
		Releases whatever   was allocated in USB_InterfaceStart

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
VOID	RTUSB_InterfaceStop(
	IN	PRTMP_ADAPTER	pAd)
{
	ASSERT(pAd);

	RTUSB_FreeDeviceDescriptor(pAd);
}

/*
	========================================================================

	Routine Description:
		Creates and submits a URB (URB_FUNCTION_RESET_PIPE) to USBD to
		reset a pipe.   If the reset pipe fails then it calls USB_ResetDevice

	Arguments:
		Adapter     Pointer to VNET_ADAPTER structure
		PipeHandle  The handle of the pipe to be reset

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_ResetPipe(
	IN	PRTMP_ADAPTER	pAd,
	IN	UINT			BulkPipe)
{
	NTSTATUS	ntStatus;
	URB			Urb;

	DBGPRINT_RAW(RT_DEBUG_ERROR, ("-->USB_ResetPipe\n"));
	ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RESET_PIPE_IN_PROGRESS);
	// Make sure there is no control message pending
	while (1)
	{
		NdisAcquireSpinLock(&pAd->ControlLock);
		if (pAd->ControlPending == FALSE)
		{
			pAd->ControlPending = TRUE;
			NdisReleaseSpinLock(&pAd->ControlLock);
			break;
		}
		else
		{
			NdisReleaseSpinLock(&pAd->ControlLock);
			NdisMSleep(500);
		}
	}

	Urb.UrbHeader.Length          = sizeof(struct _URB_PIPE_REQUEST);
	Urb.UrbHeader.Function        = URB_FUNCTION_RESET_PIPE;
	Urb.UrbPipeRequest.PipeHandle = ((BulkPipe == 0) ? (pAd->BulkInPipeHandle) : (pAd->BulkOutPipeHandle[BulkPipe - 1]));

	ntStatus = RTUSB_CallUSBD(pAd, &Urb);
	NdisAcquireSpinLock(&pAd->ControlLock);
	pAd->ControlPending = FALSE;
	NdisReleaseSpinLock(&pAd->ControlLock);

	RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RESET_PIPE_IN_PROGRESS);

	DBGPRINT_RAW(RT_DEBUG_ERROR, ("<--USB_ResetPipe\n"));

	return ntStatus;
}

/*
	========================================================================

	Routine Description:
		Creates an IRP to submite an IOCTL_INTERNAL_USB_RESET_PORT
		synchronously. Callers of this function must be running at
		PASSIVE LEVEL.

	Arguments:

	Return Value:

	IRQL =

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_ResetPort(
	IN	PRTMP_ADAPTER	pAd)
{
	NTSTATUS		ntStatus;
	PIRP			pIrp;
	KEVENT			event;
	IO_STATUS_BLOCK	ioStatus;

	DBGPRINT_RAW(RT_DEBUG_TRACE, ("--->USB_ResetDevice\n"));
	RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS);
	ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

	// issue a synchronous request
	KeInitializeEvent(&event, NotificationEvent, FALSE);

	pIrp = IoBuildDeviceIoControlRequest(
			IOCTL_INTERNAL_USB_RESET_PORT,
			pAd->pNextDeviceObject,
			NULL,
			0,
			NULL,
			0,
			TRUE,
			&event,
			&ioStatus);

	DBGPRINT_RAW(RT_DEBUG_ERROR, ("IoCallDriver(USB_ResetDevice)\n"));

	ntStatus = IoCallDriver(pAd->pNextDeviceObject, pIrp);

	if (ntStatus == STATUS_PENDING)
	{
		KeWaitForSingleObject(&event,
			Executive,
			KernelMode,
			FALSE,
			NULL);

		ntStatus = ioStatus.Status;
	}

	DBGPRINT_RAW(RT_DEBUG_TRACE, ("<---USB_ResetDevice (0x%08x)\n", ntStatus));

	RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS);

	return ntStatus;
}

/*
	========================================================================

	Routine Description:
		Indicates that all outstanding requests for a pipe should be canceled.
		This general-purpose request enables a client to cancel any pending
		transfers for the specified pipe. Pipe state and endpoint state are
		unaffected. The abort request might complete before all outstanding
		requests have completed. Do not assume that completion of the abort
		request implies that all other outstanding requests have completed.

	Arguments:
		pAd          Adpater pointer
		PipeId       0: BulkIn, 1 ~ 4: Bulk out for RT73.

	Return Value:

	IRQL = PASSIVE

	Note:

	========================================================================
*/
NTSTATUS RTUSB_AbortPipe(
	IN	PRTMP_ADAPTER	pAd,
	IN  ULONG           PipeId)
{
	NTSTATUS          ntStatus;
	URB               Urb;

	ASSERT(PipeId <= 5);
	ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

	DBGPRINT(RT_DEBUG_TRACE, ("--->RTUSB_AbortPipe\n"));

	Urb.UrbHeader.Length = (USHORT) sizeof(struct _URB_PIPE_REQUEST);
	Urb.UrbHeader.Function = URB_FUNCTION_ABORT_PIPE;
	Urb.UrbPipeRequest.PipeHandle = ((PipeId == 0) ? (pAd->BulkInPipeHandle) : (pAd->BulkOutPipeHandle[PipeId - 1]));

	ntStatus = RTUSB_CallUSBD(pAd, &Urb);

	DBGPRINT(RT_DEBUG_TRACE, ("<---RTUSB_AbortPipe, ntStatus=0x%08x\n", ntStatus));

	return ntStatus;
}

/*
	========================================================================

	Routine Description:
		This I/O request queries the status of the PDO.

	Arguments:
		pAd          Pointer to Adpater
		PortStatus   Can be one or both of USBD_PORT_ENABLED, USBD_PORT_CONNECTED

	Return Value:

	IRQL = PASSIVE

	Note:

	========================================================================
*/
NTSTATUS RTUSB_GetPortStatus(
	IN	PRTMP_ADAPTER	pAd,
	IN OUT PULONG     PortStatus)
{
	NTSTATUS			ntStatus = STATUS_SUCCESS;
	PIRP				pIrp = NULL;
	KEVENT				IoEvent;
	IO_STATUS_BLOCK		IoStatus;
	PIO_STACK_LOCATION	pNextStack;

	DBGPRINT(RT_DEBUG_TRACE, ("--->RTUSB_GetPortStatus\n"));

	// Check for correct level
	ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

	*PortStatus = 0;

	// Initialize event
	KeInitializeEvent(&IoEvent, NotificationEvent, FALSE);

	// Build an IRP in which to pass the URB to USBD
	pIrp = IoBuildDeviceIoControlRequest(
				IOCTL_INTERNAL_USB_GET_PORT_STATUS,
				pAd->pNextDeviceObject,	// Points to the next-lower
				NULL,							// No Input
				0,
				NULL,							// No output
				0,
				TRUE,							// Internal use IRP_MJ_INTERNAL_DEVICE_CONTROL
				&IoEvent,
				&IoStatus);

	if (!pIrp)
	{
		DBGPRINT_RAW(RT_DEBUG_ERROR, ("RTUSB_GetPortStatus::memory alloc for irp failed\n"));
		ntStatus = STATUS_INSUFFICIENT_RESOURCES;
		return(ntStatus);
	}

	// Pass the URB to the USB driver stack
	pNextStack = IoGetNextIrpStackLocation(pIrp);
	ASSERT(pNextStack != NULL);
	pNextStack->Parameters.Others.Argument1 = PortStatus;

	ntStatus = IoCallDriver(pAd->pNextDeviceObject, pIrp);
	if (ntStatus == STATUS_PENDING)
	{
		KeWaitForSingleObject(&IoEvent, Executive, KernelMode, FALSE, NULL);
	}
	else
	{
		IoStatus.Status = ntStatus;
	}

	// USBD maps the error code for us
	ntStatus = IoStatus.Status;

	DBGPRINT(RT_DEBUG_TRACE, ("<---RTUSB_GetPortStatus, PortStatus=0x%08x\n", *PortStatus));

	return ntStatus;
}
#endif // WIN_NDIS //

#ifdef WIN_NDIS
VOID RTUSBCmdThread(
	IN OUT PVOID Context)
{
	PRTMP_ADAPTER	pAd = (PRTMP_ADAPTER)Context;
	PCmdQElmt	cmdqelmt;
	PUCHAR	pData;
	NDIS_STATUS	NdisStatus = NDIS_STATUS_SUCCESS;
	static ULONG	cnt=0;
	NTSTATUS		ntStatus;
	NDIS_802_11_AUTHENTICATION_MODE			AuthMode;

	BOOLEAN		RestartAPIsRequired = FALSE;

	DBGPRINT(RT_DEBUG_TRACE, ("Start Command Thread\n"));
	KeSetPriorityThread(KeGetCurrentThread(), LOW_REALTIME_PRIORITY);

	ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);
	while (!pAd->TerminateThreads)
	{
		KeWaitForSingleObject(&pAd->ControlEvent,
								Suspended,
								KernelMode,
								FALSE,
								NULL);

		DBGPRINT(RT_DEBUG_INFO, ("Command Thread Triggered\n"));

		while (pAd->CmdQ.size > 0)
		{
			NdisStatus = NDIS_STATUS_SUCCESS;

			NdisAcquireSpinLock(&pAd->CmdQLock);
			RTUSBDequeueCmd(&pAd->CmdQ, &cmdqelmt);
			NdisReleaseSpinLock(&pAd->CmdQLock);
			if (cmdqelmt == NULL)
				break;

			pData = cmdqelmt->buffer;

			DBGPRINT_RAW(RT_DEBUG_INFO, ("Cmd = %x\n", cmdqelmt->command));
			if ((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)) &&
				(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HALT_IN_PROGRESS)))
			{
				switch (cmdqelmt->command)
				{
					case RT_OID_CHECK_GPIO:
						{
							ULONG 	data;
							DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!! RT_OID_CHECK_GPIO !!!\n"));

							// Read GPIO pin7 as Hardware controlled radio state

							RTUSBReadMACRegister( pAd, MAC_CSR13, &data);

							if (data & 0x80)
							{
								pAd->StaCfg.bHwRadio = TRUE;
							}
							else
							{
								pAd->StaCfg.bHwRadio = FALSE;
							}

							if (pAd->StaCfg.bRadio != (pAd->StaCfg.bHwRadio && pAd->StaCfg.bSwRadio))
							{
								pAd->StaCfg.bRadio = (pAd->StaCfg.bHwRadio && pAd->StaCfg.bSwRadio);
								if (pAd->StaCfg.bRadio == TRUE)
								{
									DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!! Radio On !!!\n"));

									MlmeRadioOn(pAd);
									// Update extra information
									pAd->ExtraInfo = EXTRA_INFO_CLEAR;
								}
								else
								{
									DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!! Radio Off !!!\n"));

									MlmeRadioOff(pAd);
									// Update extra information
									pAd->ExtraInfo = HW_RADIO_OFF;
								}
							}
						}
						break;

					case RT_OID_PERIODIC_EXECUT:
						{
							UCHAR i, temp[34];//steven:debug support for MAX
							if (pAd->OpMode == OPMODE_AP)
								APMlmePeriodicExec(pAd);
							else
							STAMlmePeriodicExec(pAd);
						}
						break;

					case OID_802_11_RX_ANTENNA_SELECTED:
						{
							NDIS_802_11_ANTENNA	Antenna = *(NDIS_802_11_ANTENNA *)pData;

							if (Antenna == 0)
								pAd->Antenna.field.RxDefaultAntenna = 1;    // ant-A
							else if(Antenna == 1)
								pAd->Antenna.field.RxDefaultAntenna = 2;    // ant-B
							else
								pAd->Antenna.field.RxDefaultAntenna = 0;    // diversity

							pAd->CommonCfg.BandState = UNKNOWN_BAND;
							AsicAntennaSelect(pAd, pAd->LatchRfRegs.Channel);
							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_RX_ANTENNA_SELECTED (=%d)\n", Antenna));
						}
						break;

					case OID_802_11_TX_ANTENNA_SELECTED:
						{
							NDIS_802_11_ANTENNA	Antenna = *(NDIS_802_11_ANTENNA *)pData;

							if (Antenna == 0)
								pAd->Antenna.field.TxDefaultAntenna = 1;    // ant-A
							else if(Antenna == 1)
								pAd->Antenna.field.TxDefaultAntenna = 2;    // ant-B
							else
								pAd->Antenna.field.TxDefaultAntenna = 0;    // diversity

							pAd->CommonCfg.BandState = UNKNOWN_BAND;
							AsicAntennaSelect(pAd, pAd->LatchRfRegs.Channel);
							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_TX_ANTENNA_SELECTED (=%d)\n", Antenna));
						}
						break;

					case RT_OID_802_11_QUERY_HARDWARE_REGISTER:
						NdisStatus = RTUSBQueryHardWareRegister(pAd, pData);
						break;

					case RT_OID_802_11_SET_HARDWARE_REGISTER:
						NdisStatus = RTUSBSetHardWareRegister(pAd, pData);
						break;

					case RT_OID_MULTI_READ_MAC:
						{
							USHORT	i;
							USHORT	Offset = *((PUSHORT)pData);
							USHORT	Length = *((PUSHORT)(pData + 2));
							RTUSBMultiRead(pAd, Offset, pData + 4, Length);
						}
						break;

					case RT_OID_MULTI_WRITE_MAC:
						{
							USHORT	Offset = *((PUSHORT)pData);
							USHORT	Length = *((PUSHORT)(pData + 2));
							RTUSBMultiWrite(pAd, Offset, pData + 4, Length);
						}
						break;

					case RT_OID_USB_VENDOR_EEPROM_READ:
						{
							USHORT	i;
							USHORT	Offset = *((PUSHORT)pData);
							USHORT	Length = *((PUSHORT)(pData + 2));
							RTUSBReadEEPROM(pAd, Offset, pData + 4, Length);
						}
						break;

					case RT_OID_USB_VENDOR_EEPROM_WRITE:
						{
							USHORT	Offset = *((PUSHORT)pData);
#if 0
							USHORT	Length = *((PUSHORT)(pData + 2));
							RT2573USBWriteEEPROM(pAd, Offset, pData + 4, Length);
#else//F/W restricts the max EEPROM write size to 62 bytes.
							USHORT	Residual = *((PUSHORT)(pData + 2));
							pData += 4;
							while (Residual > 62)
							{
								RTUSBWriteEEPROM(pAd, Offset, pData, 62);
								Offset += 62;
								Residual -= 62;
								pData += 62;
							}
							RTUSBWriteEEPROM(pAd, Offset, pData, Residual);
#endif
						}
						break;

					case RT_OID_USB_VENDOR_ENTER_TESTMODE:
						RTUSB_VendorRequest(pAd,
												0,
												DEVICE_VENDOR_REQUEST_OUT,
												0x1,
												0x4,
												0x1,
												NULL,
												0);
						break;

					case RT_OID_USB_VENDOR_EXIT_TESTMODE:
						RTUSB_VendorRequest(pAd,
												0,
												DEVICE_VENDOR_REQUEST_OUT,
												0x1,
												0x4,
												0x0,
												NULL,
												0);
						break;

					case RT_OID_USB_RESET_BULK_OUT:
						DBGPRINT_RAW(RT_DEBUG_ERROR, ("RT_OID_USB_RESET_BULK_OUT\n"));
						do
						{
							UCHAR	Index;
							ULONG    portStatus;

							//
							// From DDK,
							// All transfers must be aborted or cancelled before attempting to reset the pipe.
							//
							for (Index = 1; Index < pAd->NumberOfPipes; Index++)
								RTUSB_AbortPipe(pAd, Index);

							RTUSBRejectPendingPackets(pAd); //reject all NDIS packets waiting in TX queue
							RTUSBCancelPendingBulkOutIRP(pAd);
							RTUSBCleanUpDataBulkOutQueue(pAd);

							//
							// Get port status, device might remove.
							//
							ntStatus = RTUSB_GetPortStatus(pAd, &portStatus);

							if (NT_SUCCESS(ntStatus))
							{
								if (!(portStatus & USBD_PORT_CONNECTED))
								{
									RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST);
									DBGPRINT(RT_DEBUG_ERROR, ("RTUSB_GetPortStatus:: Device not connected, portStatus=0x%08x\n", portStatus));
									break;
								}
								else if ((!(portStatus & USBD_PORT_ENABLED)) &&
									     (portStatus & USBD_PORT_CONNECTED))
								{
									ntStatus = RTUSB_ResetPort(pAd);
								}
								else if ((portStatus & USBD_PORT_ENABLED) &&
									     (portStatus & USBD_PORT_CONNECTED))
								{
									for (Index=1; Index < pAd->NumberOfPipes; Index++)
									{
										if (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
										{
											ntStatus = RTUSB_ResetPipe(pAd, Index);
											if (NT_SUCCESS(ntStatus) == TRUE)
											{
												DBGPRINT(RT_DEBUG_ERROR, ("Bulk Out Reset on Pipe[=%d] Successed\n", Index));
											}
			 								else
											{
												DBGPRINT(RT_DEBUG_ERROR, ("Bulk Out Reset on Pipe[=%d] Failed, Status=0x%x\n", Index, ntStatus));
											}
										}
									}
								}
							}

							RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET);

							for (Index = 0; Index < 4; Index++)
							{
								if(pAd->SendTxWaitQueue[Index].Number > 0)
								{
									RTMPDeQueuePacket(pAd, Index);
								}
							}

							RTUSBKickBulkOut(pAd);

						} while (FALSE);

						break;

					case RT_OID_USB_RESET_BULK_IN:
						DBGPRINT_RAW(RT_DEBUG_ERROR, ("RT_OID_USB_RESET_BULK_IN\n"));
						//
						// From DDK,
						// All transfers must be aborted or cancelled before attempting to reset the pipe.
						//
						RTUSB_AbortPipe(pAd, 0);
						while ((pAd->PendingRx > 0) && (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
						{
							if (pAd->PendingRx > 0)
							{
								DBGPRINT_RAW(RT_DEBUG_TRACE, ("BulkIn IRP Pending, pAd->PendingRx=%d!!!\n", pAd->PendingRx));
								ntStatus = RTUSB_VendorRequest(pAd,
													0,
													DEVICE_VENDOR_REQUEST_OUT,
													0x0C,
													0x0,
													0x0,
													NULL,
													0);
								if (!NT_SUCCESS(ntStatus))
								{
									DBGPRINT_RAW(RT_DEBUG_ERROR, ("Stop RX failed, ntStatus=0x%08x\n", ntStatus));
									break;
								}
							}
						}

						if (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
						{
							ULONG    portStatus;

							//
							// Get port status, device might remove.
							//
							ntStatus = RTUSB_GetPortStatus(pAd, &portStatus);

							if (NT_SUCCESS(ntStatus))
							{
								if (!(portStatus & USBD_PORT_CONNECTED))
								{
									RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST);
									DBGPRINT(RT_DEBUG_ERROR, ("RTUSB_GetPortStatus:: Device not connected, portStatus=0x%08x\n", portStatus));
									break;
								}
								else if ((!(portStatus & USBD_PORT_ENABLED)) &&
									     (portStatus & USBD_PORT_CONNECTED))
								{
									ntStatus = RTUSB_ResetPort(pAd);
								}
								else if ((portStatus & USBD_PORT_ENABLED) &&
									     (portStatus & USBD_PORT_CONNECTED))
								{
									ntStatus = RTUSB_ResetPipe(pAd, 0);
								}
							}

							if (NT_SUCCESS(ntStatus))
							{
								USHORT	temp;
								UCHAR	index;
								DBGPRINT_RAW(RT_DEBUG_ERROR, ("Bulk In Reset Successed\n"));

								RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET);
								if ((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
									(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)) &&
									(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
									(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
								{
									for (index = 0; index < pAd->CommonCfg.NumOfBulkInIRP; index ++)
									{
										RTUSBBulkReceive(pAd);
									}
								}
							}
							else
							{
								// Reset BulkIn Pipe failed
								DBGPRINT_RAW(RT_DEBUG_ERROR, ("Bulk In Reset Failed! ntStatus=0x%x\n", ntStatus));
							}
						}
						break;

					case OID_802_11_SSID:
					{
						PNDIS_802_11_SSID pSsid = (PNDIS_802_11_SSID)pData;

						// Always reset the key table as shared key table
						RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x00);

						NdisZeroMemory(pAd->CommonCfg.Ssid, MAX_LEN_OF_SSID);
						NdisMoveMemory(pAd->CommonCfg.Ssid, pSsid->Ssid, pSsid->SsidLength);
						pAd->CommonCfg.SsidLen = (UCHAR)pSsid->SsidLength;
						RestartAPIsRequired = TRUE;
					}
						break;
					case RT_OID_802_11_DESIRED_RATES:
						NdisZeroMemory(pAd->CommonCfg.DesireRate, MAX_LEN_OF_SUPPORTED_RATES);
						NdisMoveMemory(pAd->CommonCfg.DesireRate, pData, sizeof(NDIS_802_11_RATES));
						MlmeUpdateTxRates(pAd, TRUE);
						RestartAPIsRequired = TRUE;
						break;
					case RT_OID_802_11_SET_OP_MODE:
						RTMPOPModeSwitching(pAd);
						break;
					case RT_OID_802_11_AUTHENTICATION_MODE:
						{
							int i;
							AuthMode = *(PNDIS_802_11_AUTHENTICATION_MODE)pData;
							pAd->CommonCfg.AuthMode = AuthMode;
							for (i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
							{
								if (pAd->MacTab.Content[i].Valid)
								{
									pAd->MacTab.Content[i].PortSecured  = WPA_802_1X_PORT_NOT_SECURED;
								}
							}

						}
						pAd->CommonCfg.DefaultKeyId  = 0;
						if(pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
						{
							pAd->ApCfg.WpaGTKState = SETKEYS;
							pAd->ApCfg.GKeyDoneStations = pAd->MacTab.Size;
							pAd->CommonCfg.DefaultKeyId = 1;

							// 2005-02-03 enable Pairwise key table
							RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x0000000f);
						}
						else
						{
							// 2005-02-03 disable Pairwise key table
							RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x00000000);
						}
						RestartAPIsRequired = TRUE;
						DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_AUTHENTICATION_MODE (=%d) \n",pAd->CommonCfg.AuthMode));
						break;
					case RT_OID_802_11_SET_STA_CONFIG:
						{
							RT_802_11_STA_CONFIG *pStaConfig = (RT_802_11_STA_CONFIG *)pData;
							if (pStaConfig->bEnableTxBurst != pAd->CommonCfg.bEnableTxBurst)
							{
								pAd->CommonCfg.bEnableTxBurst = (pStaConfig->bEnableTxBurst == 1);
								//Currently Tx burst mode is only implemented in infrastructure mode.
								if (INFRA_ON(pAd))
								{
									if (pAd->CommonCfg.bEnableTxBurst)
									{
										//Extend slot time if any encryption method is used to give ASIC more time to do encryption/decryption during Tx burst mode.
										if (pAd->CommonCfg.WepStatus != Ndis802_11EncryptionDisabled)
										{
										// Nemo  RT2573USBWriteMACRegister_old(pAd, MAC_CSR10, 0x20);
										}
										//Set CWmin/CWmax to 0.
										// Nemo 2004    RT2573USBWriteMACRegister_old(pAd, MAC_CSR22, 0x100);
									}
									else
									{
										if (pAd->CommonCfg.WepStatus != Ndis802_11EncryptionDisabled)
											AsicSetSlotTime(pAd, (BOOLEAN)pAd->CommonCfg.bUseShortSlotTime);
									// Nemo 2004    RT2573USBWriteMACRegister_old(pAd, MAC_CSR22, 0x53);
									}
								}
							}
							//pAd->CommonCfg.EnableTurboRate = pStaConfig->EnableTurboRate;
							pAd->CommonCfg.UseBGProtection = pStaConfig->UseBGProtection;
							//                          pAd->PortCfg.UseShortSlotTime = pStaConfig->UseShortSlotTime;
							pAd->CommonCfg.bUseShortSlotTime = 1; // 2003-10-30 always SHORT SLOT capable
							if (pAd->StaCfg.AdhocMode != pStaConfig->AdhocMode)
							{
								// allow dynamic change of "USE OFDM rate or not" in ADHOC mode
								// if setting changed, need to reset current TX rate as well as BEACON frame format
								pAd->StaCfg.AdhocMode = pStaConfig->AdhocMode;
								if (pAd->StaCfg.BssType == BSS_ADHOC)
								{
									MlmeUpdateTxRates(pAd, FALSE);
									MakeIbssBeacon(pAd);
								}
							}
							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RT_OID_802_11_SET_STA_CONFIG (Burst=%d,BGprot=%d,ShortSlot=%d,Adhoc=%d,Protection=%d\n",
								pStaConfig->bEnableTxBurst,
								pStaConfig->UseBGProtection,
								pStaConfig->UseShortSlotTime,
								pStaConfig->AdhocMode,
								pAd->CommonCfg.UseBGProtection));
						}
						break;

					case RT_OID_SET_PSM_BIT_SAVE:
						MlmeSetPsmBit(pAd, PWR_SAVE);
						RTMPSendNullFrame(pAd, pAd->CommonCfg.TxRate);
						break;

					case RT_OID_SET_RADIO:
						if (pAd->StaCfg.bRadio == TRUE)
						{
							MlmeRadioOn(pAd);
							// Update extra information
							pAd->ExtraInfo = EXTRA_INFO_CLEAR;
						}
						else
						{
							MlmeRadioOff(pAd);
							// Update extra information
							pAd->ExtraInfo = SW_RADIO_OFF;
						}
						break;

					case RT_OID_RESET_FROM_ERROR:
					case RT_OID_RESET_FROM_NDIS:
						{
							UINT	i = 0;

							RTUSBRejectPendingPackets(pAd);//reject all NDIS packets waiting in TX queue
							RTUSBCleanUpDataBulkOutQueue(pAd);
							MlmeSuspend(pAd, FALSE);

							//Add code to access necessary registers here.
							//disable Rx
							RTUSBWriteMACRegister(pAd, TXRX_CSR2, 1);
							//Ask our device to complete any pending bulk in IRP.
							while ((pAd->PendingRx > 0) ||
								(pAd->BulkOutPending[0] == TRUE) ||
								(pAd->BulkOutPending[1] == TRUE) ||
								(pAd->BulkOutPending[2] == TRUE) ||
								(pAd->BulkOutPending[3] == TRUE))
							{
								if (pAd->PendingRx > 0)
								{
									DBGPRINT_RAW(RT_DEBUG_TRACE, ("BulkIn IRP Pending!!!\n"));
									RTUSB_VendorRequest(pAd,
															0,
															DEVICE_VENDOR_REQUEST_OUT,
															0x0C,
															0x0,
															0x0,
															NULL,
															0);
								}

								if ((pAd->BulkOutPending[0] == TRUE) ||
									(pAd->BulkOutPending[1] == TRUE) ||
									(pAd->BulkOutPending[2] == TRUE) ||
									(pAd->BulkOutPending[3] == TRUE))
								{
									DBGPRINT_RAW(RT_DEBUG_TRACE, ("BulkOut IRP Pending!!!\n"));
									if (i == 0)
									{
										RTUSBCancelPendingBulkOutIRP(pAd);
										i++;
									}
								}
								NdisMSleep(500000);
							}

							NICResetFromError(pAd);
							if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HARDWARE_ERROR))
							{
								RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_HARDWARE_ERROR);
							}
							if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET))
							{
								RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET);
							}
							if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET))
							{
								RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET);
							}

							RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS);

							if ((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)) &&
								(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
								(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
							{
								MlmeResume(pAd);
								//
								// Support multiple BulkIn IRP, the value on pAd->CommonCfg.NumOfBulkInIRP may be large than 1.
								//
								for (i = 0; i < pAd->CommonCfg.NumOfBulkInIRP; i ++)
								{
									RTUSBBulkReceive(pAd);
								}
								RTUSBWriteMACRegister(pAd, TXRX_CSR2, 0x7e);
							}
						}
						break;
					case RT_OID_802_11_SET_AP_CONFIG:
						{
							RT_802_11_AP_CONFIG *pApConfig = (RT_802_11_AP_CONFIG *)pData;
							pAd->CommonCfg.bEnableTxBurst          = (pApConfig->bEnableTxBurst == 1);
							pAd->ApCfg.bIsolateInterStaTraffic = (pApConfig->IsolateInterStaTraffic == 1);
							pAd->CommonCfg.UseBGProtection         = pApConfig->UseBGProtection;
							//pAd->CommonCfg.bUseShortSlotTime       = (pApConfig->UseShortSlotTime == 1);
							pAd->CommonCfg.bUseShortSlotTime       = TRUE;
							pAd->ApCfg.AgeoutTime                  = pApConfig->AgeoutTime;
							if (pAd->ApCfg.bHideSsid != (pApConfig->HideSsid == 1))
							{
								pAd->ApCfg.bHideSsid = (pApConfig->HideSsid == 1);
								// re-built BEACON frame format
								AsicDisableSync(pAd);
								APMakeBssBeacon(pAd);
								APUpdateBeaconFrame(pAd);
								AsicEnableBssSync(pAd);
							}
							APUpdateCapabilityAndErpIe(pAd);
							DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_AP_CONFIG (Burst=%d,Hotspot=%d,HideSsid=%d,BGprot=%d,ShortSlot=%d,AgeoutTime=%d\n",
								pApConfig->bEnableTxBurst,
								pApConfig->IsolateInterStaTraffic,
								pApConfig->HideSsid,
								pApConfig->UseBGProtection,
								pApConfig->UseShortSlotTime,
								pApConfig->AgeoutTime));
						}
					       break;
					case RT_OID_ASIC_ADD_SKEY:
						{
							  AsicRemoveSharedKeyEntry(
			 					pAd,
								 BSS0,
								pAd->CommonCfg.DefaultKeyId);
							  AsicAddSharedKeyEntry(
								     pAd,
							            BSS0,
							            pAd->CommonCfg.DefaultKeyId,
							            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg,
							            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key,
							            MAX_LEN_OF_SHARE_KEY,
							            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic,
							            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic);
						}
						break;
					case RT_OID_ASIC_REMOVE_PKEY:
						{
							UCHAR idx;
							idx = *((PUCHAR)pData);
							AsicRemovePairwiseKeyEntry(pAd, idx);
						}
						break;
					case OID_802_11_CONFIGURATION:
						RestartAPIsRequired = TRUE;
						break;
					case RT_OID_LINK_DOWN:
						{
							DBGPRINT_RAW(RT_DEBUG_ERROR, ("LinkDown(RT_OID_LINK_DOWN)\n"));
							LinkDown(pAd, TRUE);
						}
						break;

					case RT_OID_VENDOR_WRITE_BBP:
						{
							UCHAR	Offset, Value;
							Offset = *((PUCHAR)pData);
							Value = *((PUCHAR)(pData + 1));
							DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%02x	value = 0x%02x\n", Offset, Value));
							RTUSBWriteBBPRegister(pAd, Offset, Value);
						}
						break;

					case RT_OID_VENDOR_READ_BBP:
						{
							UCHAR	Offset = *((PUCHAR)pData);
							PUCHAR	pValue = (PUCHAR)(pData + 1);

							DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%02x\n", Offset));
							RTUSBReadBBPRegister(pAd, Offset, pValue);
							DBGPRINT_RAW(RT_DEBUG_INFO, ("value = 0x%02x\n", *pValue));
						}
						break;

					case RT_OID_VENDOR_WRITE_RF:
						{
							ULONG	Value = *((PULONG)pData);

							DBGPRINT_RAW(RT_DEBUG_INFO, ("value = 0x%08x\n", Value));
							RTUSBWriteRFRegister(pAd, Value);
						}
						break;

					case RT_OID_802_11_RESET_COUNTERS:
						{
							UCHAR	Value[22];

							RTUSBMultiRead(pAd, STA_CSR0, Value, 24);
						}
						break;

					case RT_OID_USB_VENDOR_RESET:
						RTUSB_VendorRequest(pAd,
												0,
												DEVICE_VENDOR_REQUEST_OUT,
												1,
												1,
												0,
												NULL,
												0);
						break;

					case RT_OID_USB_VENDOR_UNPLUG:
						RTUSB_VendorRequest(pAd,
											0,
											DEVICE_VENDOR_REQUEST_OUT,
											1,
											2,
											0,
											NULL,
											0);
						break;

					case RT_OID_USB_VENDOR_SWITCH_FUNCTION:

						DBGPRINT_RAW(RT_DEBUG_ERROR, ("RT_OID_VENDOR_SWITCH_FUNCTION -- NOT SUPPORT !!\n"));

#if 0
						RT2573USBWriteMACRegister_old(pAd, MAC_CSR13, 0x2121);
						RT2573USBWriteMACRegister_old(pAd, MAC_CSR14, 0x1e1e);
						RT2573USBWriteMACRegister_old(pAd, MAC_CSR1, 3);
						RT2573USBWriteMACRegister_old(pAd, PHY_CSR4, 0xf);

						RT2573USB_VendorRequest(pAd,
												0,
												DEVICE_VENDOR_REQUEST_OUT,
												1,
												3,
												0,
												NULL,
												0);
#endif
						break;

					case RT_OID_VENDOR_FLIP_IQ:
						{
							ULONG	Value1, Value2;
							RTUSBReadMACRegister(pAd, PHY_CSR5, &Value1);
							RTUSBReadMACRegister(pAd, PHY_CSR6, &Value2);
							if (*pData == 1)
							{
								DBGPRINT_RAW(RT_DEBUG_ERROR, ("I/Q Flip\n"));
								Value1 = Value1 | 0x0004;
								Value2 = Value2 | 0x0004;
							}
							else
							{
								DBGPRINT_RAW(RT_DEBUG_ERROR, ("I/Q Not Flip\n"));
								Value1 = Value1 & 0xFFFB;
								Value2 = Value2 & 0xFFFB;
							}
							RTUSBWriteMACRegister(pAd, PHY_CSR5, Value1);
							RTUSBWriteMACRegister(pAd, PHY_CSR6, Value2);
						}
						break;

					case RT_OID_UPDATE_TX_RATE:
						MlmeUpdateTxRates(pAd, FALSE);
						if (ADHOC_ON(pAd))
						{
							MakeIbssBeacon(pAd);
							AsicEnableIbssSync(pAd);    // copy to on-chip memory
						}
						break;

					case RT_OID_802_11_SET_PREAMBLE:
						{
							USHORT	Preamble = *((PUSHORT)(cmdqelmt->buffer));
							if (Preamble == Rt802_11PreambleShort)
							{
								pAd->CommonCfg.TxPreamble = Preamble;
								MlmeSetTxPreamble(pAd, Rt802_11PreambleShort);
							}
							else if ((Preamble == Rt802_11PreambleLong) || (Preamble == Rt802_11PreambleAuto))
							{
								// if user wants AUTO, initialize to LONG here, then change according to AP's
								// capability upon association.
								pAd->CommonCfg.TxPreamble = Preamble;
								MlmeSetTxPreamble(pAd, Rt802_11PreambleLong);
							}
							else
								NdisStatus = NDIS_STATUS_INVALID_DATA;
							DBGPRINT(RT_DEBUG_ERROR, ("Set::RT_OID_802_11_SET_PREAMBLE (=%d)\n", Preamble));
						}
						break;

					case OID_802_11_NETWORK_TYPE_IN_USE:
						{
							NDIS_802_11_NETWORK_TYPE	NetType = *(PNDIS_802_11_NETWORK_TYPE)(cmdqelmt->buffer);
							if (NetType == Ndis802_11DS)
								RTMPSetPhyMode(pAd, PHY_11B);
							else if (NetType == Ndis802_11OFDM24)
								RTMPSetPhyMode(pAd, PHY_11BG_MIXED);
							else if (NetType == Ndis802_11OFDM5)
								RTMPSetPhyMode(pAd, PHY_11A);
							else
								NdisStatus = NDIS_STATUS_INVALID_DATA;
							DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_NETWORK_TYPE_IN_USE (=%d)\n",NetType));
						}
						break;

					case RT_OID_802_11_SET_PHY_MODE:
						{
							ULONG	phymode = *(ULONG *)(cmdqelmt->buffer);
							RTMPSetPhyMode(pAd, phymode);
							RestartAPIsRequired = TRUE;
							DBGPRINT(RT_DEBUG_WARN, ("SetCMD::RT_OID_802_11_SET_PHY_MODE (=%x)\n", phymode));
						}
						break;

#if 0
					case	OID_802_11_WEP_STATUS:
						{
							USHORT	Value;
							NDIS_802_11_WEP_STATUS	WepStatus = *(PNDIS_802_11_WEP_STATUS)pData;

							DBGPRINT(RT_DEBUG_ERROR, ("SetCMD::66- OID_802_11_WEP_STATUS  \n"));
						break;

							if (pAd->CommonCfg.WepStatus != WepStatus)
							{
								DBGPRINT(RT_DEBUG_ERROR, ("Config Changed !!status= %x  \n", WepStatus));

								// Config has changed
								pAd->bConfigChanged = TRUE;
							}
							pAd->CommonCfg.WepStatus = WepStatus;

#if 1
							if ((WepStatus == Ndis802_11Encryption1Enabled) &&
								(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].KeyLen != 0))
							{
								if (pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].KeyLen <= 5)
								{
									DBGPRINT(RT_DEBUG_ERROR, ("WEP64!  \n"));

									pAd->CommonCfg.CipherAlg = CIPHER_WEP64;
								}
								else
								{
									DBGPRINT(RT_DEBUG_ERROR, ("WEP128!  \n"));

									pAd->CommonCfg.CipherAlg = CIPHER_WEP128;
								}
#if 0
								RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
								Value &= 0xfe00;
								Value |= ((LENGTH_802_11 << 3) | (pAd->CommonCfg.CipherAlg));
								RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
#endif
							}
							else if (WepStatus == Ndis802_11Encryption2Enabled)
							{
								DBGPRINT(RT_DEBUG_ERROR, (" TKIP !!!  \n"));

								pAd->CommonCfg.CipherAlg = CIPHER_TKIP;
#if 0
								RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
								Value &= 0xfe00;
								Value |= ((LENGTH_802_11 << 3) | (pAd->CommonCfg.CipherAlg));
								RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
#endif
							}
							else if (WepStatus == Ndis802_11Encryption3Enabled)
							{
								DBGPRINT(RT_DEBUG_ERROR, (" AES  !!!  \n"));
								pAd->CommonCfg.CipherAlg = CIPHER_AES;
#if 0
								RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
								Value &= 0xfe00;
								Value |= ((LENGTH_802_11 << 3) | (pAd->CommonCfg.CipherAlg));
								RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
#endif
							}
							else if (WepStatus == Ndis802_11EncryptionDisabled)
							{
								DBGPRINT(RT_DEBUG_ERROR, (" CIPHER_NONE  !!!  \n"));

								pAd->CommonCfg.CipherAlg = CIPHER_NONE;
#if 0
								RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
								Value &= 0xfe00;
								RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
#endif
							}else
							{
								DBGPRINT(RT_DEBUG_ERROR, (" ERROR Cipher   !!!  \n"));
							}
#endif
						}
						break;
#endif
					case OID_802_11_ADD_WEP:
						{
							UINT	i;
							ULONG	KeyIdx;
							PNDIS_802_11_WEP	pWepKey;

							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_WEP  \n"));

							pWepKey = (PNDIS_802_11_WEP)pData;
							KeyIdx = pWepKey->KeyIndex & 0x0fffffff;

							// it is a shared key
							if ((KeyIdx >= 4) || ((pWepKey->KeyLength != 5) && (pWepKey->KeyLength != 13)))
							{
								NdisStatus = NDIS_STATUS_INVALID_DATA;
								DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_ADD_WEP, INVALID_DATA!!\n"));
							}
							else
							{
								UCHAR CipherAlg;
								pAd->SharedKey[BSS0][KeyIdx].KeyLen = (UCHAR) pWepKey->KeyLength;
								NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].Key, &pWepKey->KeyMaterial, pWepKey->KeyLength);
								CipherAlg = (pAd->SharedKey[BSS0][KeyIdx].KeyLen == 5)? CIPHER_WEP64 : CIPHER_WEP128;

								//
								// Change the WEP cipher to CKIP cipher if CKIP KP on.
								// Funk UI or Meetinghouse UI will add ckip key from this path.
								//

								if (CKIP_KP_ON(pAd))
								{
									if (CipherAlg == CIPHER_WEP64)
										CipherAlg = CIPHER_CKIP64;
									else
									 	CipherAlg = CIPHER_CKIP128;
								}

								pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CipherAlg;
								if (pWepKey->KeyIndex & 0x80000000)
								{
									// Default key for tx (shared key)
									pAd->CommonCfg.DefaultKeyId = (UCHAR) KeyIdx;
								}

								AsicAddSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx, CipherAlg, pAd->SharedKey[BSS0][KeyIdx].Key, pAd->SharedKey[BSS0][KeyIdx].KeyLen, NULL, NULL);
								DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_WEP (KeyIdx=%d, Len=%d-byte)\n", KeyIdx, pWepKey->KeyLength));
							}
						}
						break;

					case OID_802_11_REMOVE_WEP:
						{
							ULONG		KeyIdx;


							KeyIdx = *(NDIS_802_11_KEY_INDEX *) pData;
							if (KeyIdx & 0x80000000)
							{
								NdisStatus = NDIS_STATUS_INVALID_DATA;
								DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, INVALID_DATA!!\n"));
							}
							else
							{
								KeyIdx = KeyIdx & 0x0fffffff;
								if (KeyIdx >= 4)
								{
									NdisStatus = NDIS_STATUS_INVALID_DATA;
									DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, Invalid KeyIdx[=%d]!!\n", KeyIdx));
								}
								else
								{
									pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
									pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
									AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
									AsicRemovePairwiseKeyEntry(pAd, 0);
									DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_REMOVE_WEP (KeyIdx=%d)\n", KeyIdx));
								}
							}
						}
						break;

					case OID_802_11_ADD_KEY_WEP:
						{
							PNDIS_802_11_KEY		pKey;
							ULONG					i, KeyIdx;

							pKey = (PNDIS_802_11_KEY) pData;
							KeyIdx = pKey->KeyIndex & 0x0fffffff;

							 // it is a shared key
							 if (KeyIdx >= 4)
							 {
								 NdisStatus = NDIS_STATUS_INVALID_DATA;
								 DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_ADD_KEY_WEP, Invalid KeyIdx[=%d]!!\n", KeyIdx));
							 }
							 else
							 {
								 UCHAR CipherAlg;

								 pAd->SharedKey[BSS0][KeyIdx].KeyLen = (UCHAR) pKey->KeyLength;
								 NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].Key, &pKey->KeyMaterial, pKey->KeyLength);

								 if (pKey->KeyLength == 5)
									 CipherAlg = CIPHER_WEP64;
								 else
									 CipherAlg = CIPHER_WEP128;

								 if (CKIP_KP_ON(pAd))
								 {
									 if (CipherAlg == CIPHER_WEP64)
									 	CipherAlg = CIPHER_CKIP64;
									 else
									 	CipherAlg = CIPHER_CKIP128;
								 }

								 // always expand the KEY to 16-byte here for efficiency sake. so that in case CKIP is used
								 // sometime later we don't have to do key expansion for each TX in RTUSBHardTransmit().
								 // However, we shouldn't change pAd->SharedKey[BSS0][KeyIdx].KeyLen
								 if (pKey->KeyLength < 16)
								 {
									 for(i = 1; i < (16 / pKey->KeyLength); i++)
									 {
										 NdisMoveMemory(&pAd->SharedKey[BSS0][KeyIdx].Key[i * pKey->KeyLength],
														&pKey->KeyMaterial[0],
														pKey->KeyLength);
									 }
									 NdisMoveMemory(&pAd->SharedKey[BSS0][KeyIdx].Key[i * pKey->KeyLength],
													&pKey->KeyMaterial[0],
													16 - (i * pKey->KeyLength));
								 }

								 pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CipherAlg;
								 if (pKey->KeyIndex & 0x80000000)
								 {
									 // Default key for tx (shared key)
									 pAd->CommonCfg.DefaultKeyId = (UCHAR) KeyIdx;
								 }

								 AsicAddSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx, CipherAlg, pAd->SharedKey[BSS0][KeyIdx].Key, pAd->SharedKey[BSS0][KeyIdx].KeyLen, NULL, NULL);
								 DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_KEY_WEP (KeyIdx=%d, KeyLen=%d, CipherAlg=%d)\n", pAd->CommonCfg.DefaultKeyId, pAd->SharedKey[BSS0][KeyIdx].KeyLen, pAd->SharedKey[BSS0][KeyIdx].CipherAlg));
							}
						}
						break;

					case OID_802_11_ADD_KEY:
						{
							NdisStatus = RTMPWPAAddKeyProc(pAd, pData);
							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_KEY\n"));
						}
						break;

#if 0
					case RT_OID_802_11_REMOVE_WEP:
					case OID_802_11_REMOVE_WEP:
						{
							ULONG  KeyIdx;


							KeyIdx = *(NDIS_802_11_KEY_INDEX *) pData;
							if (KeyIdx & 0x80000000)
							{
								NdisStatus = NDIS_STATUS_INVALID_DATA;
								DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, INVALID_DATA!!\n"));
							}
							else
							{
								KeyIdx = KeyIdx & 0x0fffffff;
								if (KeyIdx >= 4)
								{
									NdisStatus = NDIS_STATUS_INVALID_DATA;
									DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, INVALID_DATA!!\n"));
								}
								else
								{

								}
							}
						}
						break;
#if 0
						{
							//PNDIS_802_11_REMOVE_KEY  pRemoveKey;
							ULONG  KeyIdx;
							//pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pData;
							//KeyIdx = pRemoveKey->KeyIndex;


							DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_REMOVE_WEP\n"));
							//if (InformationBufferLength != sizeof(NDIS_802_11_KEY_INDEX))
							//	Status = NDIS_STATUS_INVALID_LENGTH;
							//else
							{
								KeyIdx = *(NDIS_802_11_KEY_INDEX *) pData;

								if (KeyIdx & 0x80000000)
								{
									// Should never set default bit when remove key
									//Status = NDIS_STATUS_INVALID_DATA;
								}
								else
								{
									KeyIdx = KeyIdx & 0x0fffffff;
									if (KeyIdx >= 4)
									{
										//Status = NDIS_STATUS_INVALID_DATA;
									}
									else
									{
										pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
										//Status = RT2573USBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_WEP, TRUE, pInformationBuffer, InformationBufferLength);

										AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
										AsicRemovePairwiseKeyEntry(pAd, 0);
									}
								}
							}
						}
						break;
#endif
#endif
					case OID_802_11_REMOVE_KEY:
						{
							PNDIS_802_11_REMOVE_KEY  pRemoveKey;
							ULONG  KeyIdx;

							pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pData;
							if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
							{
								NdisStatus = RTMPWPARemoveKeyProc(pAd, pData);
								DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RTMPWPARemoveKeyProc\n"));
							}
							else
							{
								KeyIdx = pRemoveKey->KeyIndex;

								if (KeyIdx & 0x80000000)
								{
									// Should never set default bit when remove key
									NdisStatus = NDIS_STATUS_INVALID_DATA;
									DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_KEY, Invalid KeyIdx[=%d]!!\n", KeyIdx));
								}
								else
								{
									KeyIdx = KeyIdx & 0x0fffffff;
									if (KeyIdx >= 4)
									{
										NdisStatus = NDIS_STATUS_INVALID_DATA;
										DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_KEY, Invalid KeyIdx[=%d]!!\n", KeyIdx));
									}
									else
									{
										pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
										pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
										AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
										AsicRemovePairwiseKeyEntry(pAd, 0);
										DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::AsicRemoveSharedKeyEntry(KeyIdx=%d)\n", KeyIdx));
									}
								}
							}

						}
						break;


#if 0
						{
							PNDIS_802_11_REMOVE_KEY  pRemoveKey;
							ULONG  KeyIdx;
							pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pData;
							KeyIdx = pRemoveKey->KeyIndex;
							DBGPRINT(RT_DEBUG_ERROR, ("OID_802_11_REMOVE_KEY \n"));//steven:for debug

							if (pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
							{
								//Status = RT2573USBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_KEY, TRUE, pInformationBuffer, InformationBufferLength);
								NdisStatus = RTMPWPARemoveKeyProc(pAd, pData);
								AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
								AsicRemovePairwiseKeyEntry(pAd, 0);
							}
							else
							{
								// WEP
								if (KeyIdx & 0x80000000)
								{
									// Should never set default bit when remove key
									//Status = NDIS_STATUS_INVALID_DATA;
								}
								else
								{
									KeyIdx = KeyIdx & 0x0fffffff;
									if (KeyIdx >= 4)
									{
										//Status = NDIS_STATUS_INVALID_DATA;
									}
									else
									{
										pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
										//Status = RT2573USBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_KEY, TRUE, pInformationBuffer, InformationBufferLength);

										AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
										AsicRemovePairwiseKeyEntry(pAd, 0);
									}
								}
							}
						}
						break;
#endif
					case OID_802_11_POWER_MODE:
						{
							NDIS_802_11_POWER_MODE PowerMode = *(PNDIS_802_11_POWER_MODE) pData;
							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_POWER_MODE (=%d)\n",PowerMode));

							// save user's policy here, but not change StaCfg.Psm immediately
							if (PowerMode == Ndis802_11PowerModeCAM)
							{
								// clear PSM bit immediately
								MlmeSetPsmBit(pAd, PWR_ACTIVE);

								OPSTATUS_SET_FLAG(pAd, fOP_STATUS_RECEIVE_DTIM);
								if (pAd->StaCfg.bWindowsACCAMEnable == FALSE)
									pAd->StaCfg.WindowsPowerMode = PowerMode;
								pAd->StaCfg.WindowsBatteryPowerMode = PowerMode;
							}
							else if (PowerMode == Ndis802_11PowerModeMAX_PSP)
							{
								// do NOT turn on PSM bit here, wait until MlmeCheckPsmChange()
								// to exclude certain situations.
								//     MlmeSetPsmBit(pAd, PWR_SAVE);
								if (pAd->StaCfg.bWindowsACCAMEnable == FALSE)
									pAd->StaCfg.WindowsPowerMode = PowerMode;
								pAd->StaCfg.WindowsBatteryPowerMode = PowerMode;
								OPSTATUS_SET_FLAG(pAd, fOP_STATUS_RECEIVE_DTIM);
								pAd->StaCfg.DefaultListenCount = 5;
							}
							else if (PowerMode == Ndis802_11PowerModeFast_PSP)
							{
								// do NOT turn on PSM bit here, wait until MlmeCheckPsmChange()
								// to exclude certain situations.
								//     MlmeSetPsmBit(pAd, PWR_SAVE);
								OPSTATUS_SET_FLAG(pAd, fOP_STATUS_RECEIVE_DTIM);
								if (pAd->StaCfg.bWindowsACCAMEnable == FALSE)
									pAd->StaCfg.WindowsPowerMode = PowerMode;
								pAd->StaCfg.WindowsBatteryPowerMode = PowerMode;
								pAd->StaCfg.DefaultListenCount = 3;
							}
						}
						break;

					case  RT_PERFORM_SOFT_DIVERSITY:
							AsicRxAntEvalAction(pAd);
						break;

					case RT_OID_FORCE_WAKE_UP:
						AsicForceWakeup(pAd);
						break;

					case RT_OID_SET_PSM_BIT_ACTIVE:
						MlmeSetPsmBit(pAd, PWR_ACTIVE);
						break;

					case RT_OID_802_11_SET_APSD_PSM:
						pAd->CommonCfg.bAPSDForcePowerSave = *(BOOLEAN *)pData;
						if (pAd->CommonCfg.bAPSDForcePowerSave != pAd->StaCfg.Psm)
						{
							MlmeSetPsmBit(pAd, pAd->CommonCfg.bAPSDForcePowerSave);
							// Driver needs to notify AP when PSM changes
							RTMPSendNullFrame(pAd, pAd->CommonCfg.TxRate);
						}
						DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RT_OID_802_11_SET_APSD_PSM (%d)\n", pAd->CommonCfg.bAPSDForcePowerSave));
						break;

					default:
						DBGPRINT(RT_DEBUG_ERROR, ("--> Control Thread !! ERROR !! default ???? \n"));

						break;
				} //switch
			} //if (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
			else
			{
				DBGPRINT(RT_DEBUG_TRACE, ("Device not connect or Halt in progress\n"));
			}

			if (cmdqelmt->command == RT_OID_RESET_FROM_NDIS)
			{
//				DecrementIoCount(pAd);
				NdisMResetComplete(pAd->AdapterHandle, NdisStatus, FALSE);
			}

			if (cmdqelmt->CmdFromNdis == TRUE)
			{
				KIRQL oldirql;
				if ( (cmdqelmt->SetOperation) &&  (cmdqelmt->command == OID_802_11_ADD_KEY) && (pAd->StaCfg.WhqlTest == FALSE))
				{
					KeRaiseIrql(DISPATCH_LEVEL, &oldirql);
					NdisMSetInformationComplete(pAd->AdapterHandle, NdisStatus);
					KeLowerIrql(oldirql);
				}
				else if (cmdqelmt->SetOperation == FALSE)
				{

//					DecrementIoCount(pAd);
					KeRaiseIrql(DISPATCH_LEVEL, &oldirql);

					NdisMQueryInformationComplete(pAd->AdapterHandle, NdisStatus);

					KeLowerIrql(oldirql);
				}

				if ((cmdqelmt->command != RT_OID_MULTI_READ_MAC) &&
					(cmdqelmt->command != RT_OID_VENDOR_READ_BBP) &&
					(cmdqelmt->command != RT_OID_802_11_QUERY_HARDWARE_REGISTER) &&
					(cmdqelmt->command != RT_OID_USB_VENDOR_EEPROM_READ))
				{
					if (cmdqelmt->buffer != NULL)
						NdisFreeMemory(cmdqelmt->buffer, cmdqelmt->bufferlength, 0);
				}

				NdisFreeMemory(cmdqelmt, sizeof(CmdQElmt), 0);
			}
			else
			{
				if ((cmdqelmt->buffer != NULL) && (cmdqelmt->bufferlength != 0))
					NdisFreeMemory(cmdqelmt->buffer, cmdqelmt->bufferlength, 0);

				NdisFreeMemory(cmdqelmt, sizeof(CmdQElmt), 0);
			}
		}//if (Adapter->StationState != STATION_STATE_POWER_DOWN)



		// important parameter is changed, restart AP
		if (RestartAPIsRequired  && (pAd->OpMode == OPMODE_AP))
		{
			RestartAPIsRequired = FALSE;

			DBGPRINT(RT_DEBUG_WARN, ("restart AP\n"));
			if ((pAd->CommonCfg.bIEEE80211H == 1) && (pAd->CommonCfg.PhyMode == PHY_11A) && RadarChannelCheck(pAd, pAd->CommonCfg.Channel))
			{
				if (pAd->CommonCfg.RadarDetect.RDMode == RD_SILENCE_MODE)
				{
					RadarDetectionStop(pAd);
				}
				pAd->CommonCfg.RadarDetect.RDMode = RD_SWITCHING_MODE;
				pAd->CommonCfg.RadarDetect.CSCount = 0;
				DBGPRINT(RT_DEBUG_TRACE, ("APSetInformation: Channel change to ch%d, will do Channel-Switch-Announcement\n", pAd->CommonCfg.Channel));
			}
			else
			{
				DBGPRINT(RT_DEBUG_WARN, ("restart AP and re initial ap\n"));
				if (pAd->CommonCfg.RadarDetect.RDMode == RD_SILENCE_MODE)
				{
					RadarDetectionStop(pAd);
				}
				pAd->CommonCfg.RadarDetect.RDMode = RD_NORMAL_MODE;
				pAd->CommonCfg.RadarDetect.CSCount = 0;

				APStop(pAd);
				APStartUp(pAd);
			}
		}


		DBGPRINT(RT_DEBUG_INFO, ("Command Thread Finished\n"));
	}//while (!Adapter->TerminateThreads)

	pAd->hControlThread = NULL;

	DBGPRINT_RAW(RT_DEBUG_INFO, ("+ + + + Control Thread Terminated + + + + \n"));

	//DecrementIoCount(pAd);

	PsTerminateSystemThread(STATUS_SUCCESS);
}
#endif

VOID CMDHandler(
	IN PRTMP_ADAPTER pAd)
{
	PCmdQElmt cmdqelmt;
	PUCHAR pData;
	NDIS_STATUS NdisStatus = NDIS_STATUS_SUCCESS;
//	static ULONG cnt=0;
//	NTSTATUS ntStatus;
	NDIS_802_11_AUTHENTICATION_MODE AuthMode;

	unsigned long IrqFlags;

	BOOLEAN		RestartAPIsRequired = FALSE;

	while(pAd->CmdQ.size > 0)
	{
		NdisStatus = NDIS_STATUS_SUCCESS;

		NdisAcquireSpinLock(&pAd->CmdQLock,  IrqFlags);
		RTUSBDequeueCmd(&pAd->CmdQ, &cmdqelmt);
		NdisReleaseSpinLock(&pAd->CmdQLock, IrqFlags);
		if(cmdqelmt == NULL)
			break;

		pData = cmdqelmt->buffer;

		DBGPRINT_RAW(RT_DEBUG_INFO, ("Cmd = %x\n", cmdqelmt->command));
		if((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)) &&
			(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HALT_IN_PROGRESS)))
		{
			switch(cmdqelmt->command)
			{
			case RT_OID_CHECK_GPIO:
				{
					ULONG   data;
					DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!! RT_OID_CHECK_GPIO !!!\n"));

					// Read GPIO pin7 as Hardware controlled radio state

					RTUSBReadMACRegister( pAd, MAC_CSR13, &data);

					if(data & 0x80)
					{
						pAd->StaCfg.bHwRadio = TRUE;
					} else
					{
						pAd->StaCfg.bHwRadio = FALSE;
					}

					if(pAd->StaCfg.bRadio != (pAd->StaCfg.bHwRadio && pAd->StaCfg.bSwRadio))
					{
						pAd->StaCfg.bRadio = (pAd->StaCfg.bHwRadio && pAd->StaCfg.bSwRadio);
						if(pAd->StaCfg.bRadio == TRUE)
						{
							DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!! Radio On !!!\n"));

							MlmeRadioOn(pAd);
							// Update extra information
							pAd->ExtraInfo = EXTRA_INFO_CLEAR;
						} else
						{
							DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!! Radio Off !!!\n"));

							MlmeRadioOff(pAd);
							// Update extra information
							pAd->ExtraInfo = HW_RADIO_OFF;
						}
					}
				}
				break;

			case RT_OID_PERIODIC_EXECUT:
				{
//					UCHAR i, temp[34];//steven:debug support for MAX
					if(pAd->OpMode == OPMODE_AP)
						APMlmePeriodicExec(pAd);
					else
						STAMlmePeriodicExec(pAd);
				}
				break;

            case RT_OID_TBTT_EXECUT:
                pAd->ApCfg.bTimUpdate = 0;
            	pAd->ApCfg.bErpIEChange = FALSE;
            	APUpdateBeaconFrame(pAd);
                break;

			case OID_802_11_RX_ANTENNA_SELECTED:
				{
					NDIS_802_11_ANTENNA Antenna = *(NDIS_802_11_ANTENNA *)pData;

					if(Antenna == 0)
						pAd->Antenna.field.RxDefaultAntenna = 1;	// ant-A
					else if(Antenna == 1)
						pAd->Antenna.field.RxDefaultAntenna = 2;	// ant-B
					else
						pAd->Antenna.field.RxDefaultAntenna	= 0;	// diversity

					pAd->CommonCfg.BandState = UNKNOWN_BAND;
					AsicAntennaSelect(pAd, pAd->LatchRfRegs.Channel);
					DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_RX_ANTENNA_SELECTED (=%d)\n", Antenna));
				}
				break;

			case OID_802_11_TX_ANTENNA_SELECTED:
				{
					NDIS_802_11_ANTENNA Antenna = *(NDIS_802_11_ANTENNA *)pData;

					if(Antenna == 0)
						pAd->Antenna.field.TxDefaultAntenna = 1;	// ant-A
					else if(Antenna == 1)
						pAd->Antenna.field.TxDefaultAntenna = 2;	// ant-B
					else
						pAd->Antenna.field.TxDefaultAntenna	= 0;	// diversity

					pAd->CommonCfg.BandState = UNKNOWN_BAND;
					AsicAntennaSelect(pAd, pAd->LatchRfRegs.Channel);
					DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_TX_ANTENNA_SELECTED (=%d)\n", Antenna));
				}
				break;

			case RT_OID_802_11_QUERY_HARDWARE_REGISTER:
				NdisStatus = RTUSBQueryHardWareRegister(pAd, pData);
				break;

			case RT_OID_802_11_SET_HARDWARE_REGISTER:
				NdisStatus = RTUSBSetHardWareRegister(pAd, pData);
				break;

			case RT_OID_MULTI_READ_MAC:
				{
//					USHORT	i;
					USHORT  Offset = *((PUSHORT)pData);
					USHORT  Length = *((PUSHORT)(pData + 2));
					RTUSBMultiRead(pAd, Offset, pData + 4, Length);
				}
				break;

			case RT_OID_MULTI_WRITE_MAC:
				{
					USHORT  Offset = *((PUSHORT)pData);
					USHORT  Length = *((PUSHORT)(pData + 2));
					RTUSBMultiWrite(pAd, Offset, pData + 4, Length);
				}
				break;

			case RT_OID_USB_VENDOR_EEPROM_READ:
				{
//					USHORT	i;
					USHORT  Offset = *((PUSHORT)pData);
					USHORT  Length = *((PUSHORT)(pData + 2));
					RTUSBReadEEPROM(pAd, Offset, pData + 4, Length);
				}
				break;

			case RT_OID_USB_VENDOR_EEPROM_WRITE:
				{
					USHORT  Offset = *((PUSHORT)pData);
#if 0
					USHORT  Length = *((PUSHORT)(pData + 2));
					RT2573USBWriteEEPROM(pAd, Offset, pData + 4, Length);
#else//F/W restricts the max EEPROM write size to 62 bytes.
					USHORT  Residual = *((PUSHORT)(pData + 2));
					pData += 4;
					while(Residual > 62)
					{
						RTUSBWriteEEPROM(pAd, Offset, pData, 62);
						Offset += 62;
						Residual -= 62;
						pData += 62;
					}
					RTUSBWriteEEPROM(pAd, Offset, pData, Residual);
#endif
				}
				break;

			case RT_OID_USB_VENDOR_ENTER_TESTMODE:
				RTUSB_VendorRequest(pAd,
					0,
					DEVICE_VENDOR_REQUEST_OUT,
					0x1,
					0x4,
					0x1,
					NULL,
					0);
				break;

			case RT_OID_USB_VENDOR_EXIT_TESTMODE:
				RTUSB_VendorRequest(pAd,
					0,
					DEVICE_VENDOR_REQUEST_OUT,
					0x1,
					0x4,
					0x0,
					NULL,
					0);
				break;

			case RT_OID_USB_RESET_BULK_OUT:
				DBGPRINT_RAW(RT_DEBUG_ERROR, ("RT_OID_USB_RESET_BULK_OUT\n"));
				do
				{
					UCHAR   Index;
#ifdef WIN_NDIS
					ULONG    portStatus;
#endif

					//
					// From DDK,
					// All transfers must be aborted or cancelled before attempting to reset the pipe.
					//
#ifdef WIN_NDIS
					for(Index = 1; Index < pAd->NumberOfPipes; Index++)
						RTUSB_AbortPipe(pAd, Index);
#endif // WIN_NDIS //
					RTUSBRejectPendingPackets(pAd);	//reject all NDIS packets waiting in TX queue
					RTUSBCancelPendingBulkOutIRP(pAd);
					RTUSBCleanUpDataBulkOutQueue(pAd);
#ifdef WIN_NDIS
					//
					// Get port status, device might remove.
					//
					ntStatus = RTUSB_GetPortStatus(pAd, &portStatus);

					if(NT_SUCCESS(ntStatus))
					{
						if(!(portStatus & USBD_PORT_CONNECTED))
						{
							RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST);
							DBGPRINT(RT_DEBUG_ERROR, ("RTUSB_GetPortStatus:: Device not connected, portStatus=0x%08x\n", portStatus));
							break;
						} else if((!(portStatus & USBD_PORT_ENABLED)) &&
							(portStatus & USBD_PORT_CONNECTED))
						{
							ntStatus = RTUSB_ResetPort(pAd);
						} else if((portStatus & USBD_PORT_ENABLED) &&
							(portStatus & USBD_PORT_CONNECTED))
						{
							for(Index=1; Index < pAd->NumberOfPipes; Index++)
							{
								if(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
								{
									ntStatus = RTUSB_ResetPipe(pAd, Index);
									if(NT_SUCCESS(ntStatus) == TRUE)
									{
										DBGPRINT(RT_DEBUG_ERROR, ("Bulk Out Reset on Pipe[=%d] Successed\n", Index));
									} else
									{
										DBGPRINT(RT_DEBUG_ERROR, ("Bulk Out Reset on Pipe[=%d] Failed, Status=0x%x\n", Index, ntStatus));
									}
								}
							}
						}
					}
#else // WIN_NDIS //
					NICInitializeAsic(pAd);
					ReleaseAdapter(pAd, FALSE, TRUE);	// unlink urb releated tx context
					NICInitTransmit(pAd);
#endif // WIN_NDIS //
					RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RESET_PIPE_IN_PROGRESS);

					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET))
					{
						RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET);
					}

					if(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF))
					{
						for(Index = 0; Index < 4; Index++)
						{
							if(pAd->SendTxWaitQueue[Index].Number > 0)
							{
								RTMPDeQueuePacket(pAd, Index);
							}
						}

						RTUSBKickBulkOut(pAd);
					}
				} while(FALSE);
				break;

			case RT_OID_USB_RESET_BULK_IN:
				{
#ifdef WIN_NDIS
					int PendingRx;
					DBGPRINT_RAW(RT_DEBUG_ERROR, ("RT_OID_USB_RESET_BULK_IN\n"));
					//
					// From DDK,
					// All transfers must be aborted or cancelled before attempting to reset the pipe.
					//
					RTUSB_AbortPipe(pAd, 0);
					while((pAd->PendingRx > 0) && (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
					{
						if(pAd->PendingRx > 0)
						{
							DBGPRINT_RAW(RT_DEBUG_TRACE, ("BulkIn IRP Pending, pAd->PendingRx=%d!!!\n", pAd->PendingRx));
							ntStatus = RTUSB_VendorRequest(pAd,
								0,
								DEVICE_VENDOR_REQUEST_OUT,
								0x0C,
								0x0,
								0x0,
								NULL,
								0);
							if(!NT_SUCCESS(ntStatus))
							{
								DBGPRINT_RAW(RT_DEBUG_ERROR, ("Stop RX failed, ntStatus=0x%08x\n", ntStatus));
								break;
							}
						}
					}

					if(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
					{
						ULONG    portStatus;

						//
						// Get port status, device might remove.
						//
						ntStatus = RTUSB_GetPortStatus(pAd, &portStatus);

						if(NT_SUCCESS(ntStatus))
						{
							if(!(portStatus & USBD_PORT_CONNECTED))
							{
								RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST);
								DBGPRINT(RT_DEBUG_ERROR, ("RTUSB_GetPortStatus:: Device not connected, portStatus=0x%08x\n", portStatus));
								break;
							} else if((!(portStatus & USBD_PORT_ENABLED)) &&
								(portStatus & USBD_PORT_CONNECTED))
							{
								ntStatus = RTUSB_ResetPort(pAd);
							} else if((portStatus & USBD_PORT_ENABLED) &&
								(portStatus & USBD_PORT_CONNECTED))
							{
								ntStatus = RTUSB_ResetPipe(pAd, 0);
							}
						}

						if(NT_SUCCESS(ntStatus))
						{
							USHORT  temp;
							UCHAR   index;
							DBGPRINT_RAW(RT_DEBUG_ERROR, ("Bulk In Reset Successed\n"));

							RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET);
							if((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
								(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)) &&
								(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
								(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
							{
								for(index = 0; index < pAd->CommonCfg.NumOfBulkInIRP; index ++)
								{
									RTUSBBulkReceive(pAd);
								}
							}
						} else
						{
							// Reset BulkIn Pipe failed
							DBGPRINT_RAW(RT_DEBUG_ERROR, ("Bulk In Reset Failed! ntStatus=0x%x\n", ntStatus));
						}
					}
#else
					int i;
					DBGPRINT_RAW(RT_DEBUG_ERROR, ("!!!!!RT_OID_USB_RESET_BULK_IN\n"));
					RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RESET_PIPE_IN_PROGRESS);
					NICInitializeAsic(pAd);
					//RTUSBWriteMACRegister(pAd, TXRX_CSR0, 0x025eb032); // ??
					for(i = 0; i < RX_RING_SIZE; i++)
					{
						PRX_CONTEXT  pRxContext = &(pAd->RxContext[i]);

						if(pRxContext->pUrb != NULL)
						{
							RTUSB_UNLINK_URB(pRxContext->pUrb);
							usb_free_urb(pRxContext->pUrb);
							pRxContext->pUrb = NULL;
						}
						if(pRxContext->TransferBuffer != NULL)
						{
							kfree(pRxContext->TransferBuffer);
							pRxContext->TransferBuffer = NULL;
						}
					}
					NICInitRecv(pAd);
					RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RESET_PIPE_IN_PROGRESS);
					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET))
					{
						RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET);
					}

					if(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF))
					{
						RTUSBBulkReceive(pAd);
						RTUSBWriteMACRegister(pAd, TXRX_CSR0, 0x0276b032);	// enable RX of MAC block
					}
#endif
				}
				break;

			case OID_802_11_SSID:
				{
					PNDIS_802_11_SSID pSsid = (PNDIS_802_11_SSID)pData;

					// Always reset the key table as shared key table
					RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x00);

					NdisZeroMemory(pAd->CommonCfg.Ssid, MAX_LEN_OF_SSID);
					NdisMoveMemory(pAd->CommonCfg.Ssid, pSsid->Ssid, pSsid->SsidLength);
					pAd->CommonCfg.SsidLen = (UCHAR)pSsid->SsidLength;
					RestartAPIsRequired = TRUE;
				}
				break;
			case RT_OID_802_11_DESIRED_RATES:
				NdisZeroMemory(pAd->CommonCfg.DesireRate, MAX_LEN_OF_SUPPORTED_RATES);
				NdisMoveMemory(pAd->CommonCfg.DesireRate, pData, sizeof(NDIS_802_11_RATES));
				MlmeUpdateTxRates(pAd, TRUE);
				RestartAPIsRequired = TRUE;
				break;
#if 0 /* fonchi todo. */
			case RT_OID_802_11_SET_OP_MODE:
				RTMPOPModeSwitching(pAd);
				break;
#endif
			case RT_OID_802_11_AUTHENTICATION_MODE:
				{
					int i;
					AuthMode = *(PNDIS_802_11_AUTHENTICATION_MODE)pData;
					pAd->CommonCfg.AuthMode = AuthMode;
					for(i=0; i<MAX_LEN_OF_MAC_TABLE; i++)
					{
						if(pAd->MacTab.Content[i].Valid)
						{
							pAd->MacTab.Content[i].PortSecured  = WPA_802_1X_PORT_NOT_SECURED;
						}
					}

				}
				pAd->CommonCfg.DefaultKeyId  = 0;
				if(pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
				{
					pAd->ApCfg.WpaGTKState = SETKEYS;
					pAd->ApCfg.GKeyDoneStations = pAd->MacTab.Size;
					pAd->CommonCfg.DefaultKeyId = 1;

					// 2005-02-03 enable Pairwise key table
					RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x0000000f);
				} else
				{
					// 2005-02-03 disable Pairwise key table
					RTUSBWriteMACRegister(pAd, SEC_CSR4, 0x00000000);
				}
				RestartAPIsRequired = TRUE;
				DBGPRINT(RT_DEBUG_TRACE, ("Set::OID_802_11_AUTHENTICATION_MODE (=%d) \n",pAd->CommonCfg.AuthMode));
				break;
			case RT_OID_802_11_SET_STA_CONFIG:
				{
					RT_802_11_STA_CONFIG *pStaConfig = (RT_802_11_STA_CONFIG *)pData;
					if(pStaConfig->bEnableTxBurst != pAd->CommonCfg.bEnableTxBurst)
					{
						pAd->CommonCfg.bEnableTxBurst = (pStaConfig->bEnableTxBurst == 1);
						//Currently Tx burst mode is only implemented in infrastructure mode.
						if(INFRA_ON(pAd))
						{
							if(pAd->CommonCfg.bEnableTxBurst)
							{
								//Extend slot time if any encryption method is used to give ASIC more time to do encryption/decryption during Tx burst mode.
								if(pAd->CommonCfg.WepStatus != Ndis802_11EncryptionDisabled)
								{
									// Nemo  RT2573USBWriteMACRegister_old(pAd, MAC_CSR10, 0x20);
								}
								//Set CWmin/CWmax to 0.
								// Nemo 2004    RT2573USBWriteMACRegister_old(pAd, MAC_CSR22, 0x100);
							} else
							{
								if(pAd->CommonCfg.WepStatus != Ndis802_11EncryptionDisabled)
									AsicSetSlotTime(pAd, (BOOLEAN)pAd->CommonCfg.bUseShortSlotTime);
								// Nemo 2004    RT2573USBWriteMACRegister_old(pAd, MAC_CSR22, 0x53);
							}
						}
					}
					//pAd->CommonCfg.EnableTurboRate = pStaConfig->EnableTurboRate;
					pAd->CommonCfg.UseBGProtection = pStaConfig->UseBGProtection;
					//                          pAd->PortCfg.UseShortSlotTime = pStaConfig->UseShortSlotTime;
					pAd->CommonCfg.bUseShortSlotTime = 1; // 2003-10-30 always SHORT SLOT capable
					if(pAd->StaCfg.AdhocMode != pStaConfig->AdhocMode)
					{
						// allow dynamic change of "USE OFDM rate or not" in ADHOC mode
						// if setting changed, need to reset current TX rate as well as BEACON frame format
						pAd->StaCfg.AdhocMode = pStaConfig->AdhocMode;
						if(pAd->StaCfg.BssType == BSS_ADHOC)
						{
							MlmeUpdateTxRates(pAd, FALSE);
							MakeIbssBeacon(pAd);
						}
					}
					DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RT_OID_802_11_SET_STA_CONFIG (Burst=%d,BGprot=%d,ShortSlot=%d,Adhoc=%d,Protection=%d\n",
						pStaConfig->bEnableTxBurst,
						pStaConfig->UseBGProtection,
						pStaConfig->UseShortSlotTime,
						pStaConfig->AdhocMode,
						pAd->CommonCfg.UseBGProtection));
				}
				break;

			case RT_OID_SET_PSM_BIT_SAVE:
				MlmeSetPsmBit(pAd, PWR_SAVE);
				RTMPSendNullFrame(pAd, pAd->CommonCfg.TxRate);
				break;

			case RT_OID_SET_RADIO:
				if(pAd->StaCfg.bRadio == TRUE)
				{
					MlmeRadioOn(pAd);
					// Update extra information
					pAd->ExtraInfo = EXTRA_INFO_CLEAR;
				} else
				{
					MlmeRadioOff(pAd);
					// Update extra information
					pAd->ExtraInfo = SW_RADIO_OFF;
				}
				break;

			case RT_OID_RESET_FROM_ERROR:
			case RT_OID_RESET_FROM_NDIS:
				{
					UINT    i = 0;

					RTUSBRejectPendingPackets(pAd);//reject all NDIS packets waiting in TX queue
					RTUSBCleanUpDataBulkOutQueue(pAd);
					MlmeSuspend(pAd, FALSE);

					//Add code to access necessary registers here.
					//disable Rx
					RTUSBWriteMACRegister(pAd, TXRX_CSR2, 1);
					//Ask our device to complete any pending bulk in IRP.
#ifdef WIN_NDIS
					while((pAd->PendingRx > 0) ||
#else
					while((atomic_read(&pAd->PendingRx) > 0) ||
#endif
						(pAd->BulkOutPending[0] == TRUE) ||
						(pAd->BulkOutPending[1] == TRUE) ||
						(pAd->BulkOutPending[2] == TRUE) ||
						(pAd->BulkOutPending[3] == TRUE))
					{
#ifdef WIN_NDIS
						if(pAd->PendingRx > 0)
#else
						if(atomic_read(&pAd->PendingRx) > 0)
#endif
						{
							DBGPRINT_RAW(RT_DEBUG_TRACE, ("BulkIn IRP Pending!!!\n"));
							RTUSB_VendorRequest(pAd,
								0,
								DEVICE_VENDOR_REQUEST_OUT,
								0x0C,
								0x0,
								0x0,
								NULL,
								0);
						}

						if((pAd->BulkOutPending[0] == TRUE) ||
							(pAd->BulkOutPending[1] == TRUE) ||
							(pAd->BulkOutPending[2] == TRUE) ||
							(pAd->BulkOutPending[3] == TRUE))
						{
							DBGPRINT_RAW(RT_DEBUG_TRACE, ("BulkOut IRP Pending!!!\n"));
							if(i == 0)
							{
								RTUSBCancelPendingBulkOutIRP(pAd);
								i++;
							}
						}
						NdisMSleep(500000);
					}

					NICResetFromError(pAd);
					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HARDWARE_ERROR))
					{
						RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_HARDWARE_ERROR);
					}
					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET))
					{
						RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKIN_RESET);
					}
					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET))
					{
						RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BULKOUT_RESET);
					}

					RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS);

					if((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)) &&
						(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_HALT_IN_PROGRESS)) &&
						(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST)))
					{
						MlmeResume(pAd);
						//
						// Support multiple BulkIn IRP, the value on pAd->CommonCfg.NumOfBulkInIRP may be large than 1.
						//
						for(i = 0; i < pAd->CommonCfg.NumOfBulkInIRP; i ++)
						{
							RTUSBBulkReceive(pAd);
						}
						RTUSBWriteMACRegister(pAd, TXRX_CSR2, 0x7e);
					}
				}
				break;
			case RT_OID_802_11_SET_AP_CONFIG:
				{
					RT_802_11_AP_CONFIG *pApConfig = (RT_802_11_AP_CONFIG *)pData;
					pAd->CommonCfg.bEnableTxBurst          = (pApConfig->bEnableTxBurst == 1);
					pAd->ApCfg.bIsolateInterStaTraffic = (pApConfig->IsolateInterStaTraffic == 1);
					pAd->CommonCfg.UseBGProtection         = pApConfig->UseBGProtection;
					//pAd->CommonCfg.bUseShortSlotTime       = (pApConfig->UseShortSlotTime == 1);
					pAd->CommonCfg.bUseShortSlotTime       = TRUE;
					pAd->ApCfg.AgeoutTime                  = pApConfig->AgeoutTime;
					if(pAd->ApCfg.bHideSsid != (pApConfig->HideSsid == 1))
					{
						pAd->ApCfg.bHideSsid = (pApConfig->HideSsid == 1);
						// re-built BEACON frame format
						AsicDisableSync(pAd);
						APMakeBssBeacon(pAd);
						APUpdateBeaconFrame(pAd);
						AsicEnableBssSync(pAd);
					}
					APUpdateCapabilityAndErpIe(pAd);
					DBGPRINT(RT_DEBUG_TRACE, ("Set::RT_OID_802_11_SET_AP_CONFIG (Burst=%d,Hotspot=%d,HideSsid=%d,BGprot=%d,ShortSlot=%d,AgeoutTime=%d\n",
						pApConfig->bEnableTxBurst,
						pApConfig->IsolateInterStaTraffic,
						pApConfig->HideSsid,
						pApConfig->UseBGProtection,
						pApConfig->UseShortSlotTime,
						pApConfig->AgeoutTime));
				}
				break;
			case RT_OID_ASIC_ADD_SKEY:
				{
					AsicRemoveSharedKeyEntry(
						pAd,
						BSS0,
						pAd->CommonCfg.DefaultKeyId);
					AsicAddSharedKeyEntry(
						pAd,
						BSS0,
						pAd->CommonCfg.DefaultKeyId,
						pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg,
						pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key,
						MAX_LEN_OF_SHARE_KEY,
						pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic,
						pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic);
				}
				break;
			case RT_OID_ASIC_REMOVE_PKEY:
				{
					UCHAR idx;
					idx = *((PUCHAR)pData);
					AsicRemovePairwiseKeyEntry(pAd, idx);
				}
				break;
			case OID_802_11_CONFIGURATION:
				RestartAPIsRequired = TRUE;
				break;
			case RT_OID_LINK_DOWN:
				{
					DBGPRINT_RAW(RT_DEBUG_ERROR, ("LinkDown(RT_OID_LINK_DOWN)\n"));
					LinkDown(pAd, TRUE);
				}
				break;

			case RT_OID_VENDOR_WRITE_BBP:
				{
					UCHAR   Offset, Value;
					Offset = *((PUCHAR)pData);
					Value = *((PUCHAR)(pData + 1));
					DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%02x	value = 0x%02x\n", Offset, Value));
					RTUSBWriteBBPRegister(pAd, Offset, Value);
				}
				break;

			case RT_OID_VENDOR_READ_BBP:
				{
					UCHAR   Offset = *((PUCHAR)pData);
					PUCHAR  pValue = (PUCHAR)(pData + 1);

					DBGPRINT_RAW(RT_DEBUG_INFO, ("offset = 0x%02x\n", Offset));
					RTUSBReadBBPRegister(pAd, Offset, pValue);
					DBGPRINT_RAW(RT_DEBUG_INFO, ("value = 0x%02x\n", *pValue));
				}
				break;

			case RT_OID_VENDOR_WRITE_RF:
				{
					ULONG   Value = *((PULONG)pData);

					DBGPRINT_RAW(RT_DEBUG_INFO, ("value = 0x%08x\n", Value));
					RTUSBWriteRFRegister(pAd, Value);
				}
				break;

			case RT_OID_802_11_RESET_COUNTERS:
				{
					UCHAR   Value[22];

					RTUSBMultiRead(pAd, STA_CSR0, Value, 24);
				}
				break;

			case RT_OID_USB_VENDOR_RESET:
				RTUSB_VendorRequest(pAd,
					0,
					DEVICE_VENDOR_REQUEST_OUT,
					1,
					1,
					0,
					NULL,
					0);
				break;

			case RT_OID_USB_VENDOR_UNPLUG:
				RTUSB_VendorRequest(pAd,
					0,
					DEVICE_VENDOR_REQUEST_OUT,
					1,
					2,
					0,
					NULL,
					0);
				break;

			case RT_OID_USB_VENDOR_SWITCH_FUNCTION:

				DBGPRINT_RAW(RT_DEBUG_ERROR, ("RT_OID_VENDOR_SWITCH_FUNCTION -- NOT SUPPORT !!\n"));

#if 0
				RT2573USBWriteMACRegister_old(pAd, MAC_CSR13, 0x2121);
				RT2573USBWriteMACRegister_old(pAd, MAC_CSR14, 0x1e1e);
				RT2573USBWriteMACRegister_old(pAd, MAC_CSR1, 3);
				RT2573USBWriteMACRegister_old(pAd, PHY_CSR4, 0xf);

				RT2573USB_VendorRequest(pAd,
					0,
					DEVICE_VENDOR_REQUEST_OUT,
					1,
					3,
					0,
					NULL,
					0);
#endif
				break;

			case RT_OID_VENDOR_FLIP_IQ:
				{
					ULONG   Value1, Value2;
					RTUSBReadMACRegister(pAd, PHY_CSR5, &Value1);
					RTUSBReadMACRegister(pAd, PHY_CSR6, &Value2);
					if(*pData == 1)
					{
						DBGPRINT_RAW(RT_DEBUG_ERROR, ("I/Q Flip\n"));
						Value1 = Value1 | 0x0004;
						Value2 = Value2 | 0x0004;
					} else
					{
						DBGPRINT_RAW(RT_DEBUG_ERROR, ("I/Q Not Flip\n"));
						Value1 = Value1 & 0xFFFB;
						Value2 = Value2 & 0xFFFB;
					}
					RTUSBWriteMACRegister(pAd, PHY_CSR5, Value1);
					RTUSBWriteMACRegister(pAd, PHY_CSR6, Value2);
				}
				break;

			case RT_OID_UPDATE_TX_RATE:
				MlmeUpdateTxRates(pAd, FALSE);
				if(ADHOC_ON(pAd))
				{
					MakeIbssBeacon(pAd);
					AsicEnableIbssSync(pAd);	// copy to on-chip memory
				}
				break;

			case RT_OID_802_11_SET_PREAMBLE:
				{
					USHORT  Preamble = *((PUSHORT)(cmdqelmt->buffer));
					if(Preamble == Rt802_11PreambleShort)
					{
						pAd->CommonCfg.TxPreamble = Preamble;
						MlmeSetTxPreamble(pAd, Rt802_11PreambleShort);
					} else if((Preamble == Rt802_11PreambleLong) || (Preamble == Rt802_11PreambleAuto))
					{
						// if user wants AUTO, initialize to LONG here, then change according to AP's
						// capability upon association.
						pAd->CommonCfg.TxPreamble = Preamble;
						MlmeSetTxPreamble(pAd, Rt802_11PreambleLong);
					} else
						NdisStatus = NDIS_STATUS_INVALID_DATA;
					DBGPRINT(RT_DEBUG_ERROR, ("Set::RT_OID_802_11_SET_PREAMBLE (=%d)\n", Preamble));
				}
				break;

			case OID_802_11_NETWORK_TYPE_IN_USE:
				{
					NDIS_802_11_NETWORK_TYPE    NetType = *(PNDIS_802_11_NETWORK_TYPE)(cmdqelmt->buffer);
					if(NetType == Ndis802_11DS)
						RTMPSetPhyMode(pAd, PHY_11B);
					else if(NetType == Ndis802_11OFDM24)
						RTMPSetPhyMode(pAd, PHY_11BG_MIXED);
					else if(NetType == Ndis802_11OFDM5)
						RTMPSetPhyMode(pAd, PHY_11A);
					else
						NdisStatus = NDIS_STATUS_INVALID_DATA;
					DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_NETWORK_TYPE_IN_USE (=%d)\n",NetType));
				}
				break;

			case RT_OID_802_11_SET_PHY_MODE:
				{
					ULONG   phymode = *(ULONG *)(cmdqelmt->buffer);
					RTMPSetPhyMode(pAd, phymode);
					RestartAPIsRequired = TRUE;
					DBGPRINT(RT_DEBUG_WARN, ("SetCMD::RT_OID_802_11_SET_PHY_MODE (=%x)\n", phymode));
				}
				break;

#if 0
			case    OID_802_11_WEP_STATUS:
				{
					USHORT  Value;
					NDIS_802_11_WEP_STATUS  WepStatus = *(PNDIS_802_11_WEP_STATUS)pData;

					DBGPRINT(RT_DEBUG_ERROR, ("SetCMD::66- OID_802_11_WEP_STATUS  \n"));
					break;

					if(pAd->CommonCfg.WepStatus != WepStatus)
					{
						DBGPRINT(RT_DEBUG_ERROR, ("Config Changed !!status= %x  \n", WepStatus));

						// Config has changed
						pAd->bConfigChanged = TRUE;
					}
					pAd->CommonCfg.WepStatus = WepStatus;

	#if 1
					if((WepStatus == Ndis802_11Encryption1Enabled) &&
						(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].KeyLen != 0))
					{
						if(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].KeyLen <= 5)
						{
							DBGPRINT(RT_DEBUG_ERROR, ("WEP64!  \n"));

							pAd->CommonCfg.CipherAlg = CIPHER_WEP64;
						} else
						{
							DBGPRINT(RT_DEBUG_ERROR, ("WEP128!  \n"));

							pAd->CommonCfg.CipherAlg = CIPHER_WEP128;
						}
		#if 0
						RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
						Value &= 0xfe00;
						Value |= ((LENGTH_802_11 << 3) | (pAd->CommonCfg.CipherAlg));
						RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
		#endif
					} else if(WepStatus == Ndis802_11Encryption2Enabled)
					{
						DBGPRINT(RT_DEBUG_ERROR, (" TKIP !!!  \n"));

						pAd->CommonCfg.CipherAlg = CIPHER_TKIP;
		#if 0
						RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
						Value &= 0xfe00;
						Value |= ((LENGTH_802_11 << 3) | (pAd->CommonCfg.CipherAlg));
						RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
		#endif
					} else if(WepStatus == Ndis802_11Encryption3Enabled)
					{
						DBGPRINT(RT_DEBUG_ERROR, (" AES  !!!  \n"));
						pAd->CommonCfg.CipherAlg = CIPHER_AES;
		#if 0
						RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
						Value &= 0xfe00;
						Value |= ((LENGTH_802_11 << 3) | (pAd->CommonCfg.CipherAlg));
						RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
		#endif
					} else if(WepStatus == Ndis802_11EncryptionDisabled)
					{
						DBGPRINT(RT_DEBUG_ERROR, (" CIPHER_NONE  !!!  \n"));

						pAd->CommonCfg.CipherAlg = CIPHER_NONE;
		#if 0
						RTUSBReadMACRegister_old(pAd, TXRX_CSR0, &Value);
						Value &= 0xfe00;
						RTUSBWriteMACRegister_old(pAd, TXRX_CSR0, Value);
		#endif
					} else
					{
						DBGPRINT(RT_DEBUG_ERROR, (" ERROR Cipher   !!!  \n"));
					}
	#endif
				}
				break;
#endif
			case OID_802_11_ADD_WEP:
				{
//							UINT	i;
					ULONG   KeyIdx;
					PNDIS_802_11_WEP    pWepKey;

					DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_WEP  \n"));

					pWepKey = (PNDIS_802_11_WEP)pData;
					KeyIdx = pWepKey->KeyIndex & 0x0fffffff;

					// it is a shared key
					if((KeyIdx >= 4) || ((pWepKey->KeyLength != 5) && (pWepKey->KeyLength != 13)))
					{
						NdisStatus = NDIS_STATUS_INVALID_DATA;
						DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_ADD_WEP, INVALID_DATA!!\n"));
					} else
					{
						UCHAR CipherAlg;
						pAd->SharedKey[BSS0][KeyIdx].KeyLen = (UCHAR) pWepKey->KeyLength;
						NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].Key, &pWepKey->KeyMaterial, pWepKey->KeyLength);
						CipherAlg = (pAd->SharedKey[BSS0][KeyIdx].KeyLen == 5)? CIPHER_WEP64 : CIPHER_WEP128;

						//
						// Change the WEP cipher to CKIP cipher if CKIP KP on.
						// Funk UI or Meetinghouse UI will add ckip key from this path.
						//

						if(CKIP_KP_ON(pAd))
						{
							if(CipherAlg == CIPHER_WEP64)
								CipherAlg = CIPHER_CKIP64;
							else
								CipherAlg = CIPHER_CKIP128;
						}

						pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CipherAlg;
						if(pWepKey->KeyIndex & 0x80000000)
						{
							// Default key for tx (shared key)
							pAd->CommonCfg.DefaultKeyId = (UCHAR) KeyIdx;
						}

						AsicAddSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx, CipherAlg, pAd->SharedKey[BSS0][KeyIdx].Key, pAd->SharedKey[BSS0][KeyIdx].KeyLen, NULL, NULL);
						DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_WEP (KeyIdx=%d, Len=%d-byte)\n", KeyIdx, pWepKey->KeyLength));
					}
				}
				break;

			case OID_802_11_REMOVE_WEP:
				{
					ULONG       KeyIdx;


					KeyIdx = *(NDIS_802_11_KEY_INDEX *) pData;
					if(KeyIdx & 0x80000000)
					{
						NdisStatus = NDIS_STATUS_INVALID_DATA;
						DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, INVALID_DATA!!\n"));
					} else
					{
						KeyIdx = KeyIdx & 0x0fffffff;
						if(KeyIdx >= 4)
						{
							NdisStatus = NDIS_STATUS_INVALID_DATA;
							DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, Invalid KeyIdx[=%d]!!\n", KeyIdx));
						} else
						{
							pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
							pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
							AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
							AsicRemovePairwiseKeyEntry(pAd, 0);
							DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_REMOVE_WEP (KeyIdx=%d)\n", KeyIdx));
						}
					}
				}
				break;

			case OID_802_11_ADD_KEY_WEP:
				{
					PNDIS_802_11_KEY        pKey;
					ULONG                   i, KeyIdx;

					pKey = (PNDIS_802_11_KEY) pData;
					KeyIdx = pKey->KeyIndex & 0x0fffffff;

					// it is a shared key
					if(KeyIdx >= 4)
					{
						NdisStatus = NDIS_STATUS_INVALID_DATA;
						DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_ADD_KEY_WEP, Invalid KeyIdx[=%d]!!\n", KeyIdx));
					} else
					{
						UCHAR CipherAlg;

						pAd->SharedKey[BSS0][KeyIdx].KeyLen = (UCHAR) pKey->KeyLength;
						NdisMoveMemory(pAd->SharedKey[BSS0][KeyIdx].Key, &pKey->KeyMaterial, pKey->KeyLength);

						if(pKey->KeyLength == 5)
							CipherAlg = CIPHER_WEP64;
						else
							CipherAlg = CIPHER_WEP128;

						if(CKIP_KP_ON(pAd))
						{
							if(CipherAlg == CIPHER_WEP64)
								CipherAlg = CIPHER_CKIP64;
							else
								CipherAlg = CIPHER_CKIP128;
						}

						// always expand the KEY to 16-byte here for efficiency sake. so that in case CKIP is used
						// sometime later we don't have to do key expansion for each TX in RTUSBHardTransmit().
						// However, we shouldn't change pAd->SharedKey[BSS0][KeyIdx].KeyLen
						if(pKey->KeyLength < 16)
						{
							for(i = 1; i < (16 / pKey->KeyLength); i++)
							{
								NdisMoveMemory(&pAd->SharedKey[BSS0][KeyIdx].Key[i * pKey->KeyLength],
									&pKey->KeyMaterial[0],
									pKey->KeyLength);
							}
							NdisMoveMemory(&pAd->SharedKey[BSS0][KeyIdx].Key[i * pKey->KeyLength],
								&pKey->KeyMaterial[0],
								16 - (i * pKey->KeyLength));
						}

						pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CipherAlg;
						if(pKey->KeyIndex & 0x80000000)
						{
							// Default key for tx (shared key)
							pAd->CommonCfg.DefaultKeyId = (UCHAR) KeyIdx;
						}

						AsicAddSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx, CipherAlg, pAd->SharedKey[BSS0][KeyIdx].Key, pAd->SharedKey[BSS0][KeyIdx].KeyLen, NULL, NULL);
						DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_KEY_WEP (KeyIdx=%d, KeyLen=%d, CipherAlg=%d)\n", pAd->CommonCfg.DefaultKeyId, pAd->SharedKey[BSS0][KeyIdx].KeyLen, pAd->SharedKey[BSS0][KeyIdx].CipherAlg));
					}
				}
				break;

			case OID_802_11_ADD_KEY:
				{
					NdisStatus = RTMPWPAAddKeyProc(pAd, pData);
					DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_ADD_KEY\n"));
				}
				break;

#if 0
			case RT_OID_802_11_REMOVE_WEP:
			case OID_802_11_REMOVE_WEP:
				{
					ULONG  KeyIdx;


					KeyIdx = *(NDIS_802_11_KEY_INDEX *) pData;
					if(KeyIdx & 0x80000000)
					{
						NdisStatus = NDIS_STATUS_INVALID_DATA;
						DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, INVALID_DATA!!\n"));
					} else
					{
						KeyIdx = KeyIdx & 0x0fffffff;
						if(KeyIdx >= 4)
						{
							NdisStatus = NDIS_STATUS_INVALID_DATA;
							DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_WEP, INVALID_DATA!!\n"));
						} else
						{

						}
					}
				}
				break;
	#if 0
				{
					//PNDIS_802_11_REMOVE_KEY  pRemoveKey;
					ULONG  KeyIdx;
					//pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pData;
					//KeyIdx = pRemoveKey->KeyIndex;


					DBGPRINT(RT_DEBUG_ERROR, ("Set::OID_802_11_REMOVE_WEP\n"));
					//if (InformationBufferLength != sizeof(NDIS_802_11_KEY_INDEX))
					//	Status = NDIS_STATUS_INVALID_LENGTH;
					//else
					{
						KeyIdx = *(NDIS_802_11_KEY_INDEX *) pData;

						if(KeyIdx & 0x80000000)
						{
							// Should never set default bit when remove key
							//Status = NDIS_STATUS_INVALID_DATA;
						} else
						{
							KeyIdx = KeyIdx & 0x0fffffff;
							if(KeyIdx >= 4)
							{
								//Status = NDIS_STATUS_INVALID_DATA;
							} else
							{
								pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
								//Status = RT2573USBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_WEP, TRUE, pInformationBuffer, InformationBufferLength);

								AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
								AsicRemovePairwiseKeyEntry(pAd, 0);
							}
						}
					}
				}
				break;
	#endif
#endif
			case OID_802_11_REMOVE_KEY:
				{
					PNDIS_802_11_REMOVE_KEY  pRemoveKey;
					ULONG  KeyIdx;

					pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pData;
					if(pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
					{
						NdisStatus = RTMPWPARemoveKeyProc(pAd, pData);
						DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RTMPWPARemoveKeyProc\n"));
					} else
					{
						KeyIdx = pRemoveKey->KeyIndex;

						if(KeyIdx & 0x80000000)
						{
							// Should never set default bit when remove key
							NdisStatus = NDIS_STATUS_INVALID_DATA;
							DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_KEY, Invalid KeyIdx[=%d]!!\n", KeyIdx));
						} else
						{
							KeyIdx = KeyIdx & 0x0fffffff;
							if(KeyIdx >= 4)
							{
								NdisStatus = NDIS_STATUS_INVALID_DATA;
								DBGPRINT(RT_DEBUG_ERROR, ("CmdThread::OID_802_11_REMOVE_KEY, Invalid KeyIdx[=%d]!!\n", KeyIdx));
							} else
							{
								pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
								pAd->SharedKey[BSS0][KeyIdx].CipherAlg = CIPHER_NONE;
								AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
								AsicRemovePairwiseKeyEntry(pAd, 0);
								DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::AsicRemoveSharedKeyEntry(KeyIdx=%d)\n", KeyIdx));
							}
						}
					}

				}
				break;


#if 0
				{
					PNDIS_802_11_REMOVE_KEY  pRemoveKey;
					ULONG  KeyIdx;
					pRemoveKey = (PNDIS_802_11_REMOVE_KEY) pData;
					KeyIdx = pRemoveKey->KeyIndex;
					DBGPRINT(RT_DEBUG_ERROR, ("OID_802_11_REMOVE_KEY \n"));//steven:for debug

					if(pAd->CommonCfg.AuthMode >= Ndis802_11AuthModeWPA)
					{
						//Status = RT2573USBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_KEY, TRUE, pInformationBuffer, InformationBufferLength);
						NdisStatus = RTMPWPARemoveKeyProc(pAd, pData);
						AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
						AsicRemovePairwiseKeyEntry(pAd, 0);
					} else
					{
						// WEP
						if(KeyIdx & 0x80000000)
						{
							// Should never set default bit when remove key
							//Status = NDIS_STATUS_INVALID_DATA;
						} else
						{
							KeyIdx = KeyIdx & 0x0fffffff;
							if(KeyIdx >= 4)
							{
								//Status = NDIS_STATUS_INVALID_DATA;
							} else
							{
								pAd->SharedKey[BSS0][KeyIdx].KeyLen = 0;
								//Status = RT2573USBEnqueueCmdFromNdis(pAd, OID_802_11_REMOVE_KEY, TRUE, pInformationBuffer, InformationBufferLength);

								AsicRemoveSharedKeyEntry(pAd, BSS0, (UCHAR)KeyIdx);
								AsicRemovePairwiseKeyEntry(pAd, 0);
							}
						}
					}
				}
				break;
#endif
			case OID_802_11_POWER_MODE:
				{
					NDIS_802_11_POWER_MODE PowerMode = *(PNDIS_802_11_POWER_MODE) pData;
					DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::OID_802_11_POWER_MODE (=%d)\n",PowerMode));

					// save user's policy here, but not change StaCfg.Psm immediately
					if(PowerMode == Ndis802_11PowerModeCAM)
					{
						// clear PSM bit immediately
						MlmeSetPsmBit(pAd, PWR_ACTIVE);

						OPSTATUS_SET_FLAG(pAd, fOP_STATUS_RECEIVE_DTIM);
						if(pAd->StaCfg.bWindowsACCAMEnable == FALSE)
							pAd->StaCfg.WindowsPowerMode = PowerMode;
						pAd->StaCfg.WindowsBatteryPowerMode = PowerMode;
					} else if(PowerMode == Ndis802_11PowerModeMAX_PSP)
					{
						// do NOT turn on PSM bit here, wait until MlmeCheckPsmChange()
						// to exclude certain situations.
						//     MlmeSetPsmBit(pAd, PWR_SAVE);
						if(pAd->StaCfg.bWindowsACCAMEnable == FALSE)
							pAd->StaCfg.WindowsPowerMode = PowerMode;
						pAd->StaCfg.WindowsBatteryPowerMode = PowerMode;
						OPSTATUS_SET_FLAG(pAd, fOP_STATUS_RECEIVE_DTIM);
						pAd->StaCfg.DefaultListenCount = 5;
					} else if(PowerMode == Ndis802_11PowerModeFast_PSP)
					{
						// do NOT turn on PSM bit here, wait until MlmeCheckPsmChange()
						// to exclude certain situations.
						//     MlmeSetPsmBit(pAd, PWR_SAVE);
						OPSTATUS_SET_FLAG(pAd, fOP_STATUS_RECEIVE_DTIM);
						if(pAd->StaCfg.bWindowsACCAMEnable == FALSE)
							pAd->StaCfg.WindowsPowerMode = PowerMode;
						pAd->StaCfg.WindowsBatteryPowerMode = PowerMode;
						pAd->StaCfg.DefaultListenCount = 3;
					}
				}
				break;

			case  RT_PERFORM_SOFT_DIVERSITY:
				AsicRxAntEvalAction(pAd);
				break;

			case RT_OID_FORCE_WAKE_UP:
				AsicForceWakeup(pAd);
				break;

			case RT_OID_SET_PSM_BIT_ACTIVE:
				MlmeSetPsmBit(pAd, PWR_ACTIVE);
				break;

			case RT_OID_802_11_SET_APSD_PSM:
				pAd->CommonCfg.bAPSDForcePowerSave = *(BOOLEAN *)pData;
				if(pAd->CommonCfg.bAPSDForcePowerSave != pAd->StaCfg.Psm)
				{
					MlmeSetPsmBit(pAd, pAd->CommonCfg.bAPSDForcePowerSave);
					// Driver needs to notify AP when PSM changes
					RTMPSendNullFrame(pAd, pAd->CommonCfg.TxRate);
				}
				DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RT_OID_802_11_SET_APSD_PSM (%d)\n", pAd->CommonCfg.bAPSDForcePowerSave));
				break;

			default:
				DBGPRINT(RT_DEBUG_ERROR, ("--> Control Thread !! ERROR !! default ???? \n"));

				break;
			} //switch
		} //if (!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_NIC_NOT_EXIST))
		else
		{
			DBGPRINT(RT_DEBUG_TRACE, ("Device not connect or Halt in progress\n"));
		}
#ifdef WIN_NDIS
		if(cmdqelmt->command == RT_OID_RESET_FROM_NDIS)
		{
//				DecrementIoCount(pAd);
			NdisMResetComplete(pAd->AdapterHandle, NdisStatus, FALSE);
		}
#endif // WIN_NDIS //
		if(cmdqelmt->CmdFromNdis == TRUE)
		{
#ifdef WIN_NDIS
			KIRQL oldirql;
			if( (cmdqelmt->SetOperation) &&  (cmdqelmt->command == OID_802_11_ADD_KEY) && (pAd->StaCfg.WhqlTest == FALSE))
			{
				KeRaiseIrql(DISPATCH_LEVEL, &oldirql);
				NdisMSetInformationComplete(pAd->AdapterHandle, NdisStatus);
				KeLowerIrql(oldirql);
			}
			else if(cmdqelmt->SetOperation == FALSE)
			{

//					DecrementIoCount(pAd);
				KeRaiseIrql(DISPATCH_LEVEL, &oldirql);

				NdisMQueryInformationComplete(pAd->AdapterHandle, NdisStatus);

				KeLowerIrql(oldirql);
			}
#endif // WIN_NDIS
			if((cmdqelmt->command != RT_OID_MULTI_READ_MAC) &&
				(cmdqelmt->command != RT_OID_VENDOR_READ_BBP) &&
				(cmdqelmt->command != RT_OID_802_11_QUERY_HARDWARE_REGISTER) &&
				(cmdqelmt->command != RT_OID_USB_VENDOR_EEPROM_READ))
			{
				if(cmdqelmt->buffer != NULL)
					NdisFreeMemory(cmdqelmt->buffer, cmdqelmt->bufferlength, 0);
			}

			NdisFreeMemory(cmdqelmt, sizeof(CmdQElmt), 0);
		}
		else
		{
			if((cmdqelmt->buffer != NULL) && (cmdqelmt->bufferlength != 0))
				NdisFreeMemory(cmdqelmt->buffer, cmdqelmt->bufferlength, 0);

#ifdef WIN_NDIS
			NdisFreeMemory(cmdqelmt, sizeof(CmdQElmt), 0);
#else
			cmdqelmt->InUse = FALSE;
#endif
		}
	}

	// important parameter is changed, restart AP
	if (RestartAPIsRequired  && (pAd->OpMode == OPMODE_AP))
	{
		RestartAPIsRequired = FALSE;

		DBGPRINT(RT_DEBUG_WARN, ("restart AP\n"));
		if ((pAd->CommonCfg.bIEEE80211H == 1) && (pAd->CommonCfg.PhyMode == PHY_11A) && RadarChannelCheck(pAd, pAd->CommonCfg.Channel))
		{
			if (pAd->CommonCfg.RadarDetect.RDMode == RD_SILENCE_MODE)
			{
				RadarDetectionStop(pAd);
			}
			pAd->CommonCfg.RadarDetect.RDMode = RD_SWITCHING_MODE;
			pAd->CommonCfg.RadarDetect.CSCount = 0;
			DBGPRINT(RT_DEBUG_TRACE, ("APSetInformation: Channel change to ch%d, will do Channel-Switch-Announcement\n", pAd->CommonCfg.Channel));
		}
		else
		{
			DBGPRINT(RT_DEBUG_WARN, ("restart AP and re initial ap\n"));
			if (pAd->CommonCfg.RadarDetect.RDMode == RD_SILENCE_MODE)
			{
				RadarDetectionStop(pAd);
			}
			pAd->CommonCfg.RadarDetect.RDMode = RD_NORMAL_MODE;
			pAd->CommonCfg.RadarDetect.CSCount = 0;

			APStop(pAd);
			APStartUp(pAd);
		}
	}
}


/*
	========================================================================

	Routine Description:
	  Creates an IRP to submite an IOCTL_INTERNAL_USB_RESET_PORT
	  synchronously. Callers of this function must be running at
	  PASSIVE LEVEL.

	Arguments:

	Return Value:

	Note:

	========================================================================
*/
NTSTATUS	RTUSB_ResetDevice(
	IN	PRTMP_ADAPTER	pAd)
{
	NTSTATUS		Status = TRUE;

	DBGPRINT_RAW(RT_DEBUG_TRACE, ("--->USB_ResetDevice\n"));
	//RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS);
	return Status;
}

#if 1
#define HARDWARE_MAC	0
#define HARDWARE_BBP	1
#define HARDWARE_RF		2
NDIS_STATUS RTUSBQueryHardWareRegister(
	IN	PRTMP_ADAPTER	pAdapter,
	IN	PVOID			pBuf)
{
	PRT_802_11_HARDWARE_REGISTER	pHardwareRegister;
	ULONG							Value;
	USHORT							Offset;
	UCHAR							bbpValue;
	UCHAR							bbpID;
	NDIS_STATUS						Status = NDIS_STATUS_SUCCESS;

	pHardwareRegister = (PRT_802_11_HARDWARE_REGISTER) pBuf;

	if (pHardwareRegister->HardwareType == HARDWARE_MAC)
	{
		//Check Offset is valid?
		if (pHardwareRegister->Offset > 0xF4)
			Status = NDIS_STATUS_INVALID_DATA;

		Offset = (USHORT) pHardwareRegister->Offset;
		RTUSBReadMACRegister(pAdapter, Offset, &Value);
		pHardwareRegister->Data = Value;
		DBGPRINT(RT_DEBUG_TRACE, ("MAC:Offset[0x%04x]=[0x%04x]\n", Offset, Value));
	}
	else if (pHardwareRegister->HardwareType == HARDWARE_BBP)
	{
		bbpID = (UCHAR) pHardwareRegister->Offset;

		RTUSBReadBBPRegister(pAdapter, bbpID, &bbpValue);
		pHardwareRegister->Data = bbpValue;
		DBGPRINT(RT_DEBUG_TRACE, ("BBP:ID[0x%02x]=[0x%02x]\n", bbpID, bbpValue));
	}
	else
		Status = NDIS_STATUS_INVALID_DATA;

	return Status;
}

NDIS_STATUS RTUSBSetHardWareRegister(
	IN	PRTMP_ADAPTER	pAd,
	IN	PVOID			pBuf)
{
	PRT_802_11_HARDWARE_REGISTER	pHardwareRegister;
	ULONG							Value;
	USHORT							Offset;
	UCHAR							bbpValue;
	UCHAR							bbpID;
	NDIS_STATUS						Status = NDIS_STATUS_SUCCESS;

	pHardwareRegister = (PRT_802_11_HARDWARE_REGISTER) pBuf;

	if (pHardwareRegister->HardwareType == HARDWARE_MAC)
	{
		//Check Offset is valid?
		if (pHardwareRegister->Offset > 0xF4)
			Status = NDIS_STATUS_INVALID_DATA;

		Offset = (USHORT) pHardwareRegister->Offset;
		Value = (ULONG) pHardwareRegister->Data;
		RTUSBWriteMACRegister(pAd, Offset, Value);
		DBGPRINT(RT_DEBUG_TRACE, ("CmdThread::RT_OID_802_11_SET_HARDWARE_REGISTER (MAC offset=0x%08x, data=0x%08x)\n", pHardwareRegister->Offset, pHardwareRegister->Data));

		// 2004-11-08 a special 16-byte on-chip memory is used for RaConfig to pass debugging parameters to driver
		// for debug-tuning only
		if ((pHardwareRegister->Offset >= HW_DEBUG_SETTING_BASE) &&
			(pHardwareRegister->Offset <= HW_DEBUG_SETTING_END))
		{
			// 0x2bf0: test power-saving feature
			if (pHardwareRegister->Offset == HW_DEBUG_SETTING_BASE)
			{
#if 0
				ULONG isr, imr, gimr;
				USHORT tbtt = 3;

				RTMP_IO_READ32(pAd, MCU_INT_SOURCE_CSR, &isr);
				RTMP_IO_READ32(pAd, MCU_INT_MASK_CSR, &imr);
				RTMP_IO_READ32(pAd, INT_MASK_CSR, &gimr);
				DBGPRINT(RT_DEBUG_TRACE, ("Sleep %d TBTT, 8051 IMR=%08x, ISR=%08x, MAC IMR=%08x\n", tbtt, imr, isr, gimr));
				AsicSleepThenAutoWakeup(pAd, tbtt);
#endif
			}
			// 0x2bf4: test H2M_MAILBOX. byte3: Host command, byte2: token, byte1-0: arguments
			else if (pHardwareRegister->Offset == (HW_DEBUG_SETTING_BASE + 4))
			{
				// 0x2bf4: byte0 non-zero: enable R17 tuning, 0: disable R17 tuning
				if (pHardwareRegister->Data & 0x000000ff)
				{
					pAd->BbpTuning.bEnable = TRUE;
					DBGPRINT(RT_DEBUG_TRACE,("turn on R17 tuning\n"));
				}
				else
				{
					UCHAR R17;

					pAd->BbpTuning.bEnable = FALSE;
					if (OPSTATUS_TEST_FLAG(pAd, fOP_STATUS_MEDIA_STATE_CONNECTED))
					{
						if (pAd->CommonCfg.Channel > 14)
							R17 = pAd->BbpTuning.R17LowerBoundA;
						else
							R17 = pAd->BbpTuning.R17LowerBoundG;
						RTUSBWriteBBPRegister(pAd, 17, R17);
						DBGPRINT(RT_DEBUG_TRACE,("turn off R17 tuning, restore to 0x%02x\n", R17));
					}
				}
			}
			// 0x2bf8: test ACK policy and QOS format in ADHOC mode
			else if (pHardwareRegister->Offset == (HW_DEBUG_SETTING_BASE + 8))
			{
#ifdef DBG
				PUCHAR pAckStr[4] = {"NORMAL", "NO-ACK", "NO-EXPLICIT-ACK", "BLOCK-ACK"};
#endif
				EDCA_PARM DefaultEdcaParm;

				// byte0 b1-0 means ACK POLICY - 0: normal ACK, 1: no ACK, 2:no explicit ACK, 3:BA
				pAd->CommonCfg.AckPolicy[0] = ((UCHAR)pHardwareRegister->Data & 0x02) << 5;
				pAd->CommonCfg.AckPolicy[1] = ((UCHAR)pHardwareRegister->Data & 0x02) << 5;
				pAd->CommonCfg.AckPolicy[2] = ((UCHAR)pHardwareRegister->Data & 0x02) << 5;
				pAd->CommonCfg.AckPolicy[3] = ((UCHAR)pHardwareRegister->Data & 0x02) << 5;
				DBGPRINT(RT_DEBUG_TRACE, ("ACK policy = %s\n", pAckStr[(UCHAR)pHardwareRegister->Data & 0x02]));

				// any non-ZERO value in byte1 turn on EDCA & QOS format
				if (pHardwareRegister->Data & 0x0000ff00)
				{
					NdisZeroMemory(&DefaultEdcaParm, sizeof(EDCA_PARM));
					DefaultEdcaParm.bValid = TRUE;
					DefaultEdcaParm.Aifsn[0] = 3;
					DefaultEdcaParm.Aifsn[1] = 7;
					DefaultEdcaParm.Aifsn[2] = 2;
					DefaultEdcaParm.Aifsn[3] = 2;

					DefaultEdcaParm.Cwmin[0] = 4;
					DefaultEdcaParm.Cwmin[1] = 4;
					DefaultEdcaParm.Cwmin[2] = 3;
					DefaultEdcaParm.Cwmin[3] = 2;

					DefaultEdcaParm.Cwmax[0] = 10;
					DefaultEdcaParm.Cwmax[1] = 10;
					DefaultEdcaParm.Cwmax[2] = 4;
					DefaultEdcaParm.Cwmax[3] = 3;

					DefaultEdcaParm.Txop[0]  = 0;
					DefaultEdcaParm.Txop[1]  = 0;
					DefaultEdcaParm.Txop[2]  = 96;
					DefaultEdcaParm.Txop[3]  = 48;
					AsicSetEdcaParm(pAd, &DefaultEdcaParm);
				}
				else
					AsicSetEdcaParm(pAd, NULL);
			}
			// 0x2bfc: turn ON/OFF TX aggregation
			else if (pHardwareRegister->Offset == (HW_DEBUG_SETTING_BASE + 12))
			{
				if (pHardwareRegister->Data)
					OPSTATUS_SET_FLAG(pAd, fOP_STATUS_AGGREGATION_INUSED);
				else
					OPSTATUS_CLEAR_FLAG(pAd, fOP_STATUS_AGGREGATION_INUSED);
				DBGPRINT(RT_DEBUG_TRACE, ("AGGREGATION = %d\n", OPSTATUS_TEST_FLAG(pAd, fOP_STATUS_AGGREGATION_INUSED)));
			}
			else
				Status = NDIS_STATUS_NOT_SUPPORTED;
		}
	}
	else if (pHardwareRegister->HardwareType == HARDWARE_BBP)
	{
		bbpID = (UCHAR) pHardwareRegister->Offset;
		bbpValue = (UCHAR) pHardwareRegister->Data;
		RTUSBWriteBBPRegister(pAd, bbpID, bbpValue);
		DBGPRINT(RT_DEBUG_TRACE, ("BBP:ID[0x%02x]=[0x%02x]\n", bbpID, bbpValue));
	}

	return Status;
}
#endif

