/****************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ****************************************************************************

    Module Name:
    assoc.c

    Abstract:
    Handle association related requests either from WSTA or from local MLME

    Revision History:
    Who         When          What
    --------    ----------    ----------------------------------------------
    John Chang  08-04-2003    created for 11g soft-AP
 */

#include "rt_config.h"

/*
    ==========================================================================
    Description:
        association state machine init, including state transition and timer init
    Parameters:
        S - pointer to the association state machine
    Note:
        The state machine looks like the following

                                    AP_ASSOC_IDLE
        APMT2_MLME_DISASSOC_REQ    mlme_disassoc_req_action
        APMT2_PEER_DISASSOC_REQ    peer_disassoc_action
        APMT2_PEER_ASSOC_REQ       drop
        APMT2_PEER_REASSOC_REQ     drop
        APMT2_CLS3ERR              cls3err_action
    ==========================================================================
 */
VOID APAssocStateMachineInit(
    IN  PRTMP_ADAPTER   pAd,
    IN  STATE_MACHINE *S,
    OUT STATE_MACHINE_FUNC Trans[])
{
    StateMachineInit(S, Trans, AP_MAX_ASSOC_STATE, AP_MAX_ASSOC_MSG, (STATE_MACHINE_FUNC)Drop, AP_ASSOC_IDLE, AP_ASSOC_MACHINE_BASE);

    StateMachineSetAction(S, AP_ASSOC_IDLE, APMT2_MLME_DISASSOC_REQ, (STATE_MACHINE_FUNC)APMlmeDisassocReqAction);
    StateMachineSetAction(S, AP_ASSOC_IDLE, APMT2_PEER_DISASSOC_REQ, (STATE_MACHINE_FUNC)APPeerDisassocReqAction);
    StateMachineSetAction(S, AP_ASSOC_IDLE, APMT2_PEER_ASSOC_REQ,    (STATE_MACHINE_FUNC)APPeerAssocReqAction);
    StateMachineSetAction(S, AP_ASSOC_IDLE, APMT2_PEER_REASSOC_REQ,  (STATE_MACHINE_FUNC)APPeerReassocReqAction);
//  StateMachineSetAction(S, AP_ASSOC_IDLE, APMT2_CLS3ERR,           APCls3errAction);
}

/*
    ==========================================================================
    Description:
        peer assoc req handling procedure
    Parameters:
        Adapter - Adapter pointer
        Elem - MLME Queue Element
    Pre:
        the station has been authenticated and the following information is stored
    Post  :
        -# An association response frame is generated and sent to the air
    ==========================================================================
 */
VOID APPeerAssocReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    UCHAR           Addr2[MAC_ADDR_LEN];
    HEADER_802_11   AssocRspHdr;
    USHORT          ListenInterval;
    USHORT          CapabilityInfo;
    USHORT          StatusCode = 0;
    USHORT          Aid;
    PUCHAR          pOutBuffer = NULL;
    NDIS_STATUS     NStatus;
    ULONG           FrameLen = 0;
    char            Ssid[MAX_LEN_OF_SSID];
    UCHAR           SsidLen;
    UCHAR           SupportedRatesLen;
    UCHAR           SupportedRates[MAX_LEN_OF_SUPPORTED_RATES];
    UCHAR           MaxSupportedRate = 0;
    UCHAR           i;
    MAC_TABLE_ENTRY *pEntry;
    UCHAR           RSNIE_Len;
    UCHAR           RSN_IE[AP_MAX_LEN_OF_RSNIE];
    BOOLEAN         bWmmCapable;
    ULONG           RalinkIe;

    // 1. frame sanity check
    if (! PeerAssocReqSanity(pAd, Elem->Msg, Elem->MsgLen, Addr2, &CapabilityInfo, &ListenInterval, &SsidLen, &Ssid[0], &SupportedRatesLen, &SupportedRates[0],RSN_IE, &RSNIE_Len, &bWmmCapable, &RalinkIe))
        return;

    // for hidden SSID sake, SSID in AssociateRequest should be fully verified
    if ((SsidLen != pAd->CommonCfg.SsidLen) || (NdisEqualMemory(Ssid, pAd->CommonCfg.Ssid, SsidLen)==0))
        return;

    // ignore request from unwanted STA
    if (! ApCheckAccessControlList(pAd, Addr2))
        return;

    pEntry = MacTableLookup(pAd, Addr2);
    if (!pEntry)
        return;
     // clear the previous Pairwise key table
    if(pEntry->Aid != 0)
    {
       NdisZeroMemory(&pEntry->PairwiseKey, sizeof(CIPHER_KEY));
       AsicRemovePairwiseKeyEntry(pAd, (UCHAR)pEntry->Aid);
    }
    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - receive ASSOC request from %02x:%02x:%02x:%02x:%02x:%02x\n", Addr2[0],Addr2[1],Addr2[2],Addr2[3],Addr2[4],Addr2[5]));

    // supported rates array may not be sorted. sort it and find the maximum rate
    for (i=0; i<SupportedRatesLen; i++)
    {
        if (MaxSupportedRate < (SupportedRates[i] & 0x7f))
            MaxSupportedRate = SupportedRates[i] & 0x7f;
    }

    // 2. qualify this STA's auth_asoc status in the MAC table, decide StatusCode
    StatusCode = APBuildAssociation(pAd, Addr2, CapabilityInfo, MaxSupportedRate, RSN_IE, &RSNIE_Len, bWmmCapable, RalinkIe, &Aid);

    // 3. send Association Response
    NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
    if (NStatus != NDIS_STATUS_SUCCESS)
        return;

    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - Send ASSOC response (Status=%d)...\n", StatusCode));
    Aid |= 0xc000; // 2 most significant bits should be ON
    MgtMacHeaderInit(pAd, &AssocRspHdr, SUBTYPE_ASSOC_RSP, 0, Addr2, pAd->CommonCfg.Bssid);

    MakeOutgoingFrame(pOutBuffer,               &FrameLen,
                      sizeof(HEADER_802_11),    &AssocRspHdr,
                      2,                        &CapabilityInfo,
                      2,                        &StatusCode,
                      2,                        &Aid,
                      1,                        &SupRateIe,
                      1,                        &pAd->CommonCfg.SupRateLen,
                      pAd->CommonCfg.SupRateLen,    pAd->CommonCfg.SupRate,
                      END_OF_ARGS);

    if (pAd->CommonCfg.ExtRateLen)
    {
        UCHAR ErpIeLen = 1;
        ULONG TmpLen;
        MakeOutgoingFrame(pOutBuffer+FrameLen,      &TmpLen,
                          1,                        &ErpIe,
                          1,                        &ErpIeLen,
                          1,                        &pAd->ApCfg.ErpIeContent,
                          1,                        &ExtRateIe,
                          1,                        &pAd->CommonCfg.ExtRateLen,
                          pAd->CommonCfg.ExtRateLen,    pAd->CommonCfg.ExtRate,
                          END_OF_ARGS);
        FrameLen += TmpLen;
    }
#if 0
    // add WMM IE here
    if (pAd->CommonCfg.bWmmCapable && CLIENT_STATUS_TEST_FLAG(pEntry, fCLIENT_STATUS_WMM_CAPABLE))
    {
        ULONG TmpLen;
        UCHAR WmeParmIe[26] = {IE_VENDOR_SPECIFIC, 24, 0x00, 0x50, 0xf2, 0x02, 0x01, 0x01, 0, 0};
        WmeParmIe[8] = pAd->ApCfg.BssEdcaParm.EdcaUpdateCount & 0x0f;
        for (i=QID_AC_BE; i<=QID_AC_VO; i++)
        {
            WmeParmIe[10+ (i*4)] = (i << 5)                                         +     // b5-6 is ACI
                                   ((UCHAR)pAd->ApCfg.BssEdcaParm.bACM[i] << 4)     +     // b4 is ACM
                                   (pAd->ApCfg.BssEdcaParm.Aifsn[i] & 0x0f);              // b0-3 is AIFSN
            WmeParmIe[11+ (i*4)] = (pAd->ApCfg.BssEdcaParm.Cwmax[i] << 4)           +     // b5-8 is CWMAX
                                   (pAd->ApCfg.BssEdcaParm.Cwmin[i] & 0x0f);              // b0-3 is CWMIN
            WmeParmIe[12+ (i*4)] = (UCHAR)(pAd->ApCfg.BssEdcaParm.Txop[i] & 0xff);        // low byte of TXOP
            WmeParmIe[13+ (i*4)] = (UCHAR)(pAd->ApCfg.BssEdcaParm.Txop[i] >> 8);          // high byte of TXOP
        }

        MakeOutgoingFrame(pOutBuffer+FrameLen,      &TmpLen,
                          26,                       WmeParmIe,
                          END_OF_ARGS);
        FrameLen += TmpLen;
    }

    // add Ralink-specific IE here - Byte0.b0=1 for aggregation, Byte0.b1=1 for piggy-back
    if (pAd->CommonCfg.bAggregationCapable)
    {
        ULONG TmpLen;
	 UCHAR RalinkSpecificIe[9] = {IE_VENDOR_SPECIFIC, 7, 0x00, 0x0c, 0x43, 0x01, 0x00, 0x00, 0x00};
	        MakeOutgoingFrame(pOutBuffer+FrameLen,       &TmpLen,
	                          9,                         RalinkSpecificIe,
	                          END_OF_ARGS);
	 FrameLen += TmpLen;
    }
  #endif
    MiniportMMRequest(pAd, pOutBuffer, FrameLen);

    if (((pEntry->AuthMode == Ndis802_11AuthModeWPAPSK) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK)) && (StatusCode == MLME_SUCCESS))
    {
        // Enqueue a EAPOL-start message with the pEntry
		if (!pEntry->EnqueueStartForPSKTimerRunning)
		{				
			RTMPSetTimer(&pEntry->EnqueueStartForPSKTimer, (300 * HZ)/1000);
			pEntry->EnqueueStartForPSKTimerRunning = TRUE;	
		}
    }
}

/*
    ==========================================================================
    Description:
        mlme reassoc req handling procedure
    Parameters:
        Elem -
    Pre:
        -# SSID  (Adapter->ApCfg.ssid[])
        -# BSSID (AP address, Adapter->ApCfg.bssid)
        -# Supported rates (Adapter->ApCfg.supported_rates[])
        -# Supported rates length (Adapter->ApCfg.supported_rates_len)
        -# Tx power (Adapter->ApCfg.tx_power)
    ==========================================================================
 */
VOID APPeerReassocReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    UCHAR           ApAddr[MAC_ADDR_LEN], Addr2[MAC_ADDR_LEN];
    HEADER_802_11   ReassocRspHdr;
    USHORT          CapabilityInfo, ListenInterval;
    USHORT          StatusCode = MLME_SUCCESS;
    USHORT          Aid = 1;
    ULONG           FrameLen = 0;
    NDIS_STATUS     NStatus;
    PUCHAR          pOutBuffer = NULL;
    char            Ssid[MAX_LEN_OF_SSID];
    UCHAR           SsidLen;
    UCHAR           SupportedRatesLen;
    UCHAR           SupportedRates[MAX_LEN_OF_SUPPORTED_RATES];
    UCHAR           MaxSupportedRate = 0;
    int             i;
    MAC_TABLE_ENTRY *pEntry;
    UCHAR           RSNIE_Len;
    UCHAR           RSN_IE[AP_MAX_LEN_OF_RSNIE];
    BOOLEAN         bWmmCapable;
    ULONG           RalinkIe;

    // 1. frame sanity check
    if (! PeerReassocReqSanity(pAd, Elem->Msg, Elem->MsgLen, Addr2, &CapabilityInfo, &ListenInterval, ApAddr, &SsidLen, &Ssid[0], &SupportedRatesLen, &SupportedRates[0],RSN_IE, &RSNIE_Len, &bWmmCapable, &RalinkIe))
        return;

    // for hidden SSID sake, SSID in AssociateRequest should be fully verified
    if ((SsidLen != pAd->CommonCfg.SsidLen) || (NdisEqualMemory(Ssid, pAd->CommonCfg.Ssid, SsidLen)==0))
        return;

    // ignore request from unwanted STA
    if (! ApCheckAccessControlList(pAd, Addr2))
        return;

    pEntry = MacTableLookup(pAd, Addr2);
    if (!pEntry)
        return;
     // clear the previous Pairwise key table
    if(pEntry->Aid != 0)
    {
       NdisZeroMemory(&pEntry->PairwiseKey, sizeof(CIPHER_KEY));
       AsicRemovePairwiseKeyEntry(pAd, (UCHAR)pEntry->Aid);
    }
    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - receive RE-ASSOC request from %02x:%02x:%02x:%02x:%02x:%02x\n", Addr2[0],Addr2[1],Addr2[2],Addr2[3],Addr2[4],Addr2[5]));

    // supported rates array may not be sorted. sort it and find the maximum rate
    for (i=0; i<SupportedRatesLen; i++)
    {
        if (MaxSupportedRate < (SupportedRates[i] & 0x7f))
            MaxSupportedRate = SupportedRates[i] & 0x7f;
    }

    // 2. qualify this STA's auth_asoc status in the MAC table, decide StatusCode
    StatusCode = APBuildAssociation(pAd, Addr2, CapabilityInfo, MaxSupportedRate, RSN_IE, &RSNIE_Len, bWmmCapable, RalinkIe, &Aid);

    // 3. reply Re-association Response
    NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
    if (NStatus != NDIS_STATUS_SUCCESS)
        return;

    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - Send RE-ASSOC response (Status = %d)...\n", StatusCode));
    Aid |= 0xc000; // 2 most significant bits should be ON
    MgtMacHeaderInit(pAd, &ReassocRspHdr, SUBTYPE_REASSOC_RSP, 0, Addr2, pAd->CommonCfg.Bssid);

    MakeOutgoingFrame(pOutBuffer,               &FrameLen,
                      sizeof(HEADER_802_11),    &ReassocRspHdr,
                      2,                        &CapabilityInfo,
                      2,                        &StatusCode,
                      2,                        &Aid,
                      1,                        &SupRateIe,
                      1,                        &pAd->CommonCfg.SupRateLen,
                      pAd->CommonCfg.SupRateLen,    pAd->CommonCfg.SupRate,
                      END_OF_ARGS);

    if (pAd->CommonCfg.ExtRateLen)
    {
        UCHAR ErpIeLen = 1;
        ULONG TmpLen;
        MakeOutgoingFrame(pOutBuffer+FrameLen,      &TmpLen,
                          1,                        &ErpIe,
                          1,                        &ErpIeLen,
                          1,                        &pAd->ApCfg.ErpIeContent,
                          1,                        &ExtRateIe,
                          1,                        &pAd->CommonCfg.ExtRateLen,
                          pAd->CommonCfg.ExtRateLen,    pAd->CommonCfg.ExtRate,
                          END_OF_ARGS);
        FrameLen += TmpLen;
    }

    // add WMM IE here
    if (pAd->CommonCfg.bWmmCapable && CLIENT_STATUS_TEST_FLAG(pEntry, fCLIENT_STATUS_WMM_CAPABLE))
    {
        ULONG TmpLen;
        UCHAR WmeParmIe[26] = {IE_VENDOR_SPECIFIC, 24, 0x00, 0x50, 0xf2, 0x02, 0x01, 0x01, 0, 0};
        WmeParmIe[8] = pAd->ApCfg.BssEdcaParm.EdcaUpdateCount & 0x0f;
        for (i=QID_AC_BE; i<=QID_AC_VO; i++)
        {
            WmeParmIe[10+ (i*4)] = (i << 5)                                         +     // b5-6 is ACI
                                   ((UCHAR)pAd->ApCfg.BssEdcaParm.bACM[i] << 4)     +     // b4 is ACM
                                   (pAd->ApCfg.BssEdcaParm.Aifsn[i] & 0x0f);              // b0-3 is AIFSN
            WmeParmIe[11+ (i*4)] = (pAd->ApCfg.BssEdcaParm.Cwmax[i] << 4)           +     // b5-8 is CWMAX
                                   (pAd->ApCfg.BssEdcaParm.Cwmin[i] & 0x0f);              // b0-3 is CWMIN
            WmeParmIe[12+ (i*4)] = (UCHAR)(pAd->ApCfg.BssEdcaParm.Txop[i] & 0xff);        // low byte of TXOP
            WmeParmIe[13+ (i*4)] = (UCHAR)(pAd->ApCfg.BssEdcaParm.Txop[i] >> 8);          // high byte of TXOP
        }

        MakeOutgoingFrame(pOutBuffer+FrameLen,      &TmpLen,
                          26,                       WmeParmIe,
                          END_OF_ARGS);
        FrameLen += TmpLen;
    }

    // add Ralink-specific IE here - Byte0.b0=1 for aggregation, Byte0.b1=1 for piggy-back
    if (pAd->CommonCfg.bAggregationCapable)
    {


	    ULONG TmpLen;
	    UCHAR RalinkSpecificIe[9] = {IE_VENDOR_SPECIFIC, 7, 0x00, 0x0c, 0x43, 0x01, 0x00, 0x00, 0x00};
	    MakeOutgoingFrame(pOutBuffer+FrameLen,       &TmpLen,
	                     9,                         RalinkSpecificIe,
	                      END_OF_ARGS);
	    FrameLen += TmpLen;
    }

    MiniportMMRequest(pAd, pOutBuffer, FrameLen);

    if (((pEntry->AuthMode == Ndis802_11AuthModeWPAPSK) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK)) && (StatusCode == MLME_SUCCESS))
    {
        // Enqueue a EAPOL-start message with the pEntry
		if (!pEntry->EnqueueStartForPSKTimerRunning)
		{				
			RTMPSetTimer(&pEntry->EnqueueStartForPSKTimer, (300 * HZ)/1000);
			pEntry->EnqueueStartForPSKTimerRunning = TRUE;	
		}
    }
}

/*
    ==========================================================================
    Description:
        left part of IEEE 802.11/1999 p.374
    Parameters:
        Elem - MLME message containing the received frame
    ==========================================================================
 */
VOID APPeerDisassocReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    UCHAR         Addr2[MAC_ADDR_LEN];
    USHORT        Reason;
    MAC_TABLE_ENTRY       *pEntry;

    if (! PeerDisassocReqSanity(pAd, Elem->Msg, Elem->MsgLen, Addr2, &Reason))
        return;

    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - receive DIS-ASSOC request from %02x:%02x:%02x:%02x:%02x:%02x, reason=%d\n", Addr2[0],Addr2[1],Addr2[2],Addr2[3],Addr2[4],Addr2[5],Reason));

    pEntry = MacTableLookup(pAd, Addr2);
    if (pEntry)
    {
        ApLogEvent(pAd, Addr2, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, Addr2);
    }
}


/*
    ==========================================================================
    Description:
        Upper layer orders to disassoc s STA
    Parameters:
        Elem -
    ==========================================================================
 */
VOID APMlmeDisassocReqAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
    MLME_DISASSOC_REQ_STRUCT *DisassocReq;
    HEADER_802_11        DisassocHdr;
    PUCHAR                pOutBuffer = NULL;
    ULONG                 FrameLen = 0;
    NDIS_STATUS           NStatus;
    MAC_TABLE_ENTRY       *pEntry;

    DisassocReq = (MLME_DISASSOC_REQ_STRUCT *)(Elem->Msg);

    pEntry = MacTableLookup(pAd, DisassocReq->Addr);
    if (pEntry)
    {
        ApLogEvent(pAd, DisassocReq->Addr, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, DisassocReq->Addr);
    }

    // 2. send out a DISASSOC request frame
    NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
    if (NStatus != NDIS_STATUS_SUCCESS)
        return;

    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - MLME disassociates %02x:%02x:%02x:%02x:%02x:%02x; Send DISASSOC request\n",
        DisassocReq->Addr[0],DisassocReq->Addr[1],DisassocReq->Addr[2],
        DisassocReq->Addr[3],DisassocReq->Addr[4],DisassocReq->Addr[5]));
    MgtMacHeaderInit(pAd, &DisassocHdr, SUBTYPE_DISASSOC, 0, DisassocReq->Addr, pAd->CommonCfg.Bssid);
    MakeOutgoingFrame(pOutBuffer,            &FrameLen,
                      sizeof(HEADER_802_11), &DisassocHdr,
                      2,                     &DisassocReq->Reason,
                      END_OF_ARGS);
    MiniportMMRequest(pAd, pOutBuffer, FrameLen);
}


/*
    ==========================================================================
    Description:
        right part of IEEE 802.11/1999 page 374
    Note:
        This event should never cause ASSOC state machine perform state
        transition, and has no relationship with CNTL machine. So we separate
        this routine as a service outside of ASSOC state transition table.
    ==========================================================================
 */
VOID APCls3errAction(
    IN PRTMP_ADAPTER pAd,
    IN PUCHAR        pAddr)
{
    HEADER_802_11         DisassocHdr;
    PUCHAR                pOutBuffer = NULL;
    ULONG                 FrameLen = 0;
    NDIS_STATUS           NStatus;
    USHORT                Reason = REASON_CLS3ERR;
    MAC_TABLE_ENTRY       *pEntry;

    pEntry = MacTableLookup(pAd, pAddr);
    if (pEntry)
    {
        //ApLogEvent(pAd, pAddr, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, pAddr);
    }

    // 2. send out a DISASSOC request frame
    NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
    if (NStatus != NDIS_STATUS_SUCCESS)
        return;

    DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - Class 3 Error, Send DISASSOC frame to %02x:%02x:%02x:%02x:%02x:%02x\n", pAddr[0],pAddr[1],pAddr[2],pAddr[3],pAddr[4],pAddr[5]));
    MgtMacHeaderInit(pAd, &DisassocHdr, SUBTYPE_DISASSOC, 0, pAddr, pAd->CommonCfg.Bssid);
    MakeOutgoingFrame(pOutBuffer,            &FrameLen,
                      sizeof(HEADER_802_11), &DisassocHdr,
                      2,                     &Reason,
                      END_OF_ARGS);
    MiniportMMRequest(pAd, pOutBuffer, FrameLen);
}

 /*
     ==========================================================================
     Description:
        search for entire MAC table to find next available AID to be used
        if none is available, return 0
     ==========================================================================
  */
USHORT APAssignAid(
    IN PRTMP_ADAPTER pAd,
    IN PMAC_TABLE_ENTRY pEntry)
{
    USHORT Aid = 0;
#if 0
    USHORT Aid=0, i;
    UCHAR AidUsed[MAX_LEN_OF_MAC_TABLE];

    // find all used AID
    NdisZeroMemory(AidUsed, MAX_LEN_OF_MAC_TABLE);
    for (i=0;i<MAX_LEN_OF_MAC_TABLE;i++)
        if (pAd->MacTab.Content[i].Valid)
            AidUsed[pAd->MacTab.Content[i].Aid] = 1;

    // find smallest unused AID
    for (i=1; i<MAX_LEN_OF_MAC_TABLE; i++)
        if (AidUsed[i]==0)
        {
            Aid = i;
            break;
        }

    return Aid;
#else
    if (pEntry)
        Aid = (pEntry - &pAd->MacTab.Content[0])/sizeof(MAC_TABLE_ENTRY);

    DBGPRINT(RT_DEBUG_TRACE, ("APAssignAid(pEntry=%d, AID=%d)\n", pEntry==NULL, Aid));
    return Aid;
#endif
}

/*
    ==========================================================================
    Description:
       assign a new AID to the newly associated/re-associated STA and
       decide its MaxSupportedRate and CurrTxRate. Both rates should not
       exceed AP's capapbility
    Return:
       MLME_SUCCESS - association successfully built
       others - association failed due to resource issue
    ==========================================================================
 */
USHORT APBuildAssociation(
    IN PRTMP_ADAPTER pAd,
    IN PUCHAR        pAddr,
    IN USHORT        CapabilityInfo,
    IN UCHAR         MaxSupportedRateIn500Kbps,
    IN UCHAR         *RSN,
    IN UCHAR         *pRSNLen,
    IN BOOLEAN       bWmmCapable,
    IN ULONG         ClientRalinkIe,
    OUT USHORT       *pAid)
{
    USHORT           StatusCode = MLME_SUCCESS;
    PMAC_TABLE_ENTRY pEntry;
    UCHAR            MaxSupportedRate = RATE_11;

    switch (MaxSupportedRateIn500Kbps)
    {
        case 108: MaxSupportedRate = RATE_54;   break;
        case 96:  MaxSupportedRate = RATE_48;   break;
        case 72:  MaxSupportedRate = RATE_36;   break;
        case 48:  MaxSupportedRate = RATE_24;   break;
        case 36:  MaxSupportedRate = RATE_18;   break;
        case 24:  MaxSupportedRate = RATE_12;   break;
        case 18:  MaxSupportedRate = RATE_9;    break;
        case 12:  MaxSupportedRate = RATE_6;    break;
        case 22:  MaxSupportedRate = RATE_11;   break;
        case 11:  MaxSupportedRate = RATE_5_5;  break;
        case 4:   MaxSupportedRate = RATE_2;    break;
        case 2:   MaxSupportedRate = RATE_1;    break;
        default:  MaxSupportedRate = RATE_11;   break;
    }

    if ((pAd->CommonCfg.PhyMode == PHY_11G) && (MaxSupportedRate < RATE_FIRST_OFDM_RATE))
        return MLME_ASSOC_REJ_DATA_RATE;

    pEntry = MacTableLookup(pAd, pAddr);
    if (pEntry && ((pEntry->Sst == SST_AUTH) || (pEntry->Sst == SST_ASSOC)))
    {
        // TODO:
        // should qualify other parameters, for example - capablity, supported rates, listen interval, ... etc
        // to decide the Status Code
        //*pAid = APAssignAid(pAd, pEntry);
        //pEntry->Aid = *pAid;
        *pAid = pEntry->Aid;

        NdisMoveMemory(pEntry->RSN_IE, RSN, *pRSNLen);
        pEntry->RSNIE_Len = *pRSNLen;
        DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - RSNIE_Len (0x%x) = 0x%x, RSNIE = 0x%x:%x:%x:%x\n",
            *pRSNLen, pEntry->RSNIE_Len, pEntry->RSN_IE[0], pEntry->RSN_IE[1], pEntry->RSN_IE[2], pEntry->RSN_IE[3]));

        if (*pAid == 0)
            StatusCode = MLME_ASSOC_REJ_UNABLE_HANDLE_STA;
        else
        {
            pEntry->Sst = SST_ASSOC;
            pEntry->MaxSupportedRate = min(pAd->CommonCfg.MaxTxRate, MaxSupportedRate);
            pEntry->CapabilityInfo = CapabilityInfo;
            if (pEntry->AuthMode == Ndis802_11AuthModeWPAPSK)
            {
                pEntry->PrivacyFilter = Ndis802_11PrivFilter8021xWEP;
                pEntry->WpaState = AS_INITPSK;
            }
            else if (pEntry->AuthMode >= Ndis802_11AuthModeWPA)
            {
                pEntry->PrivacyFilter = Ndis802_11PrivFilter8021xWEP;
                pEntry->WpaState = AS_AUTHENTICATION;
            }

            if (bWmmCapable)
            {
                CLIENT_STATUS_SET_FLAG(pEntry, fCLIENT_STATUS_WMM_CAPABLE);
            }
            else
            {
                CLIENT_STATUS_CLEAR_FLAG(pEntry, fCLIENT_STATUS_WMM_CAPABLE);
            }

            if (pAd->CommonCfg.bAggregationCapable)
            {
	            if (ClientRalinkIe & 0x00000001)
	            {
	             		CLIENT_STATUS_SET_FLAG(pEntry, fCLIENT_STATUS_AGGREGATION_CAPABLE);

	            }
            }
            else
            {
                CLIENT_STATUS_CLEAR_FLAG(pEntry, fCLIENT_STATUS_AGGREGATION_CAPABLE);

            }

#if 0
            // If dynamic rate switching is enabled, starts from a more reliable rate to
            // increase STA's DHCP succeed rate
            if (OPSTATUS_TEST_FLAG(pAd, fOP_STATUS_TX_RATE_SWITCH_ENABLED))
            {
                pEntry->CurrTxRate = min(pEntry->MaxSupportedRate, RATE_11);
            }
            else
#endif
                pEntry->CurrTxRate = pEntry->MaxSupportedRate;

            if (pEntry->AuthMode < Ndis802_11AuthModeWPA)
                ApLogEvent(pAd, pEntry->Addr, EVENT_ASSOCIATED);

            APUpdateCapabilityAndErpIe(pAd);

            pEntry->ReTryCounter = PEER_MSG1_RETRY_TIMER_CTR;

            DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - WMM=%d, AuthMode=%d, WepStatus=%d, WpaState=%d, GroupKeyWepStatus=%d, AGGRE=%d\n",
                bWmmCapable, pEntry->AuthMode, pEntry->WepStatus, pEntry->WpaState, pAd->CommonCfg.GroupKeyWepStatus,
                CLIENT_STATUS_TEST_FLAG(pEntry, fCLIENT_STATUS_AGGREGATION_CAPABLE)));
            DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - Assign AID=%d to STA %02x:%02x:%02x:%02x:%02x:%02x, MaxSupRate=%d Mbps, CurrTxRate=%d Mbps\n\n",
                *pAid, pAddr[0], pAddr[1], pAddr[2], pAddr[3], pAddr[4], pAddr[5],
                RateIdToMbps[pEntry->MaxSupportedRate], RateIdToMbps[pEntry->CurrTxRate]));

            StatusCode = MLME_SUCCESS;
        }
    }
    else // CLASS 3 error should have been handled beforehand; here should be MAC table full
        StatusCode = MLME_ASSOC_REJ_UNABLE_HANDLE_STA;

    return StatusCode;
}

